/* main.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/*string encoding_charset;*/
/*bool new_window_option;*/
bool _new_document;
bool _show_version;
string _client_id;
string[] _remaining_args;

const OptionEntry[] options = {
{ "client-id", '\0', 0, OptionArg.STRING, ref _client_id, N_("Specify session management ID"), N_("ID") },
{ "version", 'V', 0, OptionArg.NONE, ref _show_version, N_("Show the application's version"), null },
/* { "encoding", '\0', 0, OptionArg.STRING, ref encoding_charset, N_("Set the character encoding to be used to open the files listed on the command line"), N_("ENCODING")}, */
/*{ "new-window", '\0', 0, OptionArg.NONE, ref new_window_option, N_("Create a new toplevel window in an existing instance of valide"), null },*/
{ "new-document", '\0', 0, OptionArg.NONE, ref _new_document, N_("Create a new document in an existing instance of valide"), null },
/* "" = G_OPTION_REMAINING */
{ "", '\0', 0, OptionArg.FILENAME_ARRAY, ref _remaining_args, null, N_("[FILE...]") }, /* collects file arguments */
{null}
};

public class Valide.Application : Object
{
  private bool show_version;
  private bool new_document;
  private string[] file_list;
  private string client_id;
  private HashTable<string, Window> windows;

  public string[] args { get; construct; }

  private enum Action
  {
    NEW,
    OPEN,
    INVALID,
    ACTIVATE
  }

  private bool launch_window (Window window, Action action, string[] uris)
  {
    bool opened = false;

    switch (action)
    {
      case Action.ACTIVATE:
        window.present ();
      break;
      case Action.NEW:
        try
        {
          window.documents.create ();
          opened = true;
        }
        catch (Error e)
        {
          debug (e.message);
        }
      break;
      case Action.OPEN:
        if (uris != null)
        {
          for (int i = 0; uris[i] != null; i++)
          {
            try
            {
              if (window.projects.accept_file (uris[i]))
              {
                window.projects.open (uris[i]);
              }
              else
              {
                window.documents.create (uris[i]);
                opened = true;
              }
            }
            catch (Error e)
            {
              debug (e.message);
            }
          }
        }
      break;
    }
    return opened;
  }

  private Window create_new_window (Action action, string[] uris,
                                    Gdk.Screen? screen = null)
  {
    bool opened;
    Window window;

    window = new Window ();
    if (screen != null)
    {
      window.set_screen (screen);
    }
    opened = this.launch_window (window, action, uris);
    window.run ();
    return window;
  }

#if (UNIQUE)
  private Unique.Response message (Unique.App sender, int command,
                                   Unique.MessageData data, uint time)
  {
    string ws;
    Action action;
    Window window;
    uint workspace;
    string[] uris = null;

    switch (command)
    {
      case Unique.Command.OPEN:
        action = Action.OPEN;
        uris = data.get_uris ();
      break;
      case Unique.Command.NEW:
        action = Action.NEW;
      break;
      case Unique.Command.ACTIVATE:
        action = Action.ACTIVATE;
      break;
      case Unique.Command.INVALID:
      default:
        action = Action.INVALID;
      break;
    }

    workspace = data.get_workspace ();
    /* The first instance doesn't set the workspace */
    if (workspace == 0)
    {
      workspace = Unique.Backend.create ().get_workspace ();
    }
    ws = workspace.to_string ();
    window = this.windows.lookup (ws);
    if (window == null)
    {
      window = this.create_new_window (action, uris, data.get_screen ());
      window.workspace = workspace;
      window.destroy.connect ((s) => {
        string ws_s;

        ws_s = (s as Window).workspace.to_string ();
        this.windows.remove (ws_s);
        if (this.windows.size () == 0)
        {
          Gtk.main_quit ();
        }
      });
      this.windows.insert (ws, window);
    }
    else
    {
      this.launch_window (window, action, uris);
    }
    return Unique.Response.OK;
  }
#endif

  /* The construct function can't throws error */
  public Application (string[] args) throws OptionError
  {
    OptionContext context;

    Gtk.init (ref args);
    Intl.bindtextdomain (Config.GETTEXT_PACKAGE, Config.LOCALE_DIR);
    Intl.bind_textdomain_codeset (Config.GETTEXT_PACKAGE, "UTF-8");

    Object (args: args);

    /* Setup command line options */
    context = new OptionContext (_("- Integrated Development Environment for Vala"));
    context.add_main_entries (options, Config.GETTEXT_PACKAGE);
    context.add_group (Gtk.get_option_group (false));

    unowned string[] args2 = this.args;
    context.parse (ref args2);
    if (_remaining_args != null)
    {
      for (int i = 0; _remaining_args[i] != null; i++)
      {
        this.file_list += _remaining_args[i];
      }
    }
    this.show_version = _show_version;
    this.new_document = _new_document;
    if (_client_id != null)
    {
      this.client_id = _client_id;
    }
    else
    {
      this.client_id = "org.valide.valaide";
    }
    this.windows = new HashTable<string, Window> (str_hash, str_equal);
  }

#if (!UNIQUE)
  public int run ()
  {
    if (this.show_version)
    {
      print ("%s - Version %s\n", Environment.get_application_name (),
                                  Config.VERSION);
    }
    else
    {
      Window window;
      Action action;

      if (this.new_document)
      {
        action = Action.NEW;
      }
      else if (this.file_list != null)
      {
        action = Action.OPEN;
      }
      else
      {
        action = Action.ACTIVATE;
      }
      window = this.create_new_window (action, this.file_list);
      window.destroy.connect (() => {
        Gtk.main_quit ();
      });
      Gtk.main ();
    }
    return 0;
  }
#else
  public int run ()
  {
    Unique.App app;

    if (this.show_version)
    {
      print ("%s - Version %s\n", Environment.get_application_name (),
                                  Config.VERSION);
    }
    else
    {
      Unique.Command command;
      Unique.MessageData data;

      data = new Unique.MessageData ();
      if (this.new_document)
      {
        command = Unique.Command.NEW;
      }
      else if (this.file_list != null)
      {
        command = Unique.Command.OPEN;
        data.set_uris (this.file_list);
      }
      else
      {
        command = Unique.Command.ACTIVATE;
      }

      app = new Unique.App (this.client_id, null);
      if (app.is_running)
      {
        app.send_message (command, data);
        Gdk.notify_startup_complete ();
      }
      else
      {
        this.message (app, command, data, 0);
        app.message_received.connect (message);
        Gtk.main ();
      }
    }
    return 0;
  }
#endif
}

int main (string[] args)
{
  int status;

  try
  {
    Valide.Application application;

    application = new Valide.Application (args);
    status = application.run ();
  }
  catch (OptionError e)
  {
    print (_("%s\nRun '%s --help' to see a full list of available command line options.\n"),
           e.message, args[0]);
    status = -1;
  }
  return status;
}

