/* symbol-browser.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/*
 * This code is re-writing in the vala language of GeditSymbolBrowserPlugin
 *
 * Copyright (C) 2007 - Micah Carrick
 */

using Valide;

/* just used to keep the population code readable */
private class Symbol : Object
{
  public string name { get; set; }
  public string full_name { get; set; }
  public string signature { get; set; }
  public string[] inherits { get; set; }
  public string typeref { get; set; }
  public string access { get; set; }
  public string implementation { get; set; }
  public string kind { get; set; }
  public string file { get; set; }
  public string language { get; set; }
  public string line { get; set; }
  public Symbol root;
  public unowned Document document;
}

private errordomain SymbolError
{
  KIND
}

public class SymbolBrowser : Plugin, Object
{
  /* enum for tree model columns, not all is currently used */
  enum Col
  {
    ICON,
    NAME,
    FILE,
    LINE,
    LANGUAGE,
    NB
  }

  private int atom;
  private Gtk.VBox box;
  private Gtk.Label label;
  private uint status_context;
  private Gtk.TreeStore model;
  private Gtk.TreeView tree_view;
  private SList<string> expanded_rows;
  private Gtk.ScrolledWindow scrolled_window;
  private Gtk.TreeViewColumn language_column;
  private Gtk.TreeViewColumn source_file_column;
  private Gtk.TreeViewColumn line_number_column;
  private HashTable<string, Gtk.TreeIter?> root_nodes;

  /**
   * @see Plugin.path
   */
  public string path { get; construct set; }
  /**
   * @see Plugin.window
   */
  public Window window { get; construct set; }

  private bool show_language_column
  {
    get
    {
      return this.window.config_manager.get_boolean ("SymbolBrowser",
                                                     "show-language");
    }
    set
    {
      this.window.config_manager.set_boolean ("SymbolBrowser", "show-language",
                                              value);
      this.language_column.visible = value;
    }
  }

  private bool show_source_file_column
  {
    get
    {
      return this.window.config_manager.get_boolean ("SymbolBrowser",
                                                     "show-source-file");
    }
    set
    {
      this.window.config_manager.set_boolean ("SymbolBrowser",
                                              "show-source-file", value);
      this.source_file_column.visible = value;
    }
  }

  private bool show_line_number_column
  {
    get
    {
      return this.window.config_manager.get_boolean ("SymbolBrowser",
                                                     "show-line-number");
    }
    set
    {
      this.window.config_manager.set_boolean ("SymbolBrowser",
                                              "show-line-number", value);
      this.line_number_column.visible = value;
    }
  }

  private bool hierarchic_view
  {
    get
    {
      return this.window.config_manager.get_boolean ("SymbolBrowser",
                                                     "hierarchic-view");
    }
    set
    {
      this.window.config_manager.set_boolean ("SymbolBrowser",
                                              "hierarchic-view", value);
      this.populate_symbols ();
    }
  }

  private bool is_expanded_row (string name)
  {
    bool ret = false;

    foreach (string i in this.expanded_rows)
    {
      if (i == name)
      {
        ret = true;
        break;
      }
    }
    return ret;
  }

  private void add_expanded_row (string name)
  {
    if (!this.is_expanded_row (name))
    {
      this.expanded_rows.prepend (name);
    }
  }

  private void remove_expanded_row (string name)
  {
    unowned SList<string> l;

    if (this.is_expanded_row (name) == true)
    {
      for (l = this.expanded_rows; l != null; l = l.next)
      {
        if (l.data == name)
        {
          this.expanded_rows.remove_all (l.data);
        }
      }
    }
  }

  /**
   * Re-expand any row when the list is repopulated if that row was previously
   * expanded.
   */
  private void restore_expanded_rows ()
  {
    /* is this going to fix the crash? */
    if (this.window.documents.sensitive == true)
    {
      Gtk.TreeIter iter;
      Gtk.TreeModel model;

      model = this.tree_view.get_model ();
      if (model.get_iter_first (out iter) == true)
      {
        for (bool r = model.get_iter_first (out iter);
             r == true;
             r = model.iter_next (ref iter))
        {
          string name;

          model.get (iter, Col.NAME, out name, -1);
          if (this.is_expanded_row (name))
          {
            Gtk.TreePath path = model.get_path (iter);
            this.tree_view.expand_row (path, false);
          }
        }
      }
    }
  }

  /**
   * This function needs work. Used to be custom parsed for ex searches
   * which could potentially contain tabs, however, I've decided that isn't
   * really needed. But, it's still using Symbol. We could just pass back
   * **fields array.
   */
  private Symbol? parse_line (string line)
  {
    /* (parent.)name  file  ex_cmd;"  kind:[enumvalue|errorcode|errordomain|enum|struct|namespace]  line:x  language:x  access:x */
    /* (parent.)name  file  ex_cmd;"  kind:field  line:x  language:x  typeref:x  access:x */
    /* (parent.)name  file  ex_cmd;"  kind:class  line:x  language:x  inherits:x  access:x  implementation:x */
    /* (parent.)name  file  ex_cmd;"  kind:[delegate|method|signal]  line:x  language:x  typeref:x  access:x  implementation:x  signature:x */
    /* (parent.)name  file  ex_cmd;"  kind:interface  line:x  language:x  inherits:x  access:x */
    /* (parent.)name  file  ex_cmd;"  kind:property  line:x  language:x  typeref:x  access:x  implementation:x */
    /*          name  file  ex_cmd;"  kind:function  line:x  language:x  signature:x */
    /*          name  file  ex_cmd;"  kind:variable  line:x  language:x */

    int name_ext;
    Symbol symbol = null;
    string[] fields = null;

    fields = line.split ("\t", 0);
    if (fields[0] != null)
    {
      symbol = new Symbol ();
      symbol.full_name = fields[0];
      name_ext = symbol.full_name.last_index_of_char ('.');
      if (name_ext != -1)
      {
        symbol.name = symbol.full_name.substring (name_ext + 1);
      }
      else
      {
        symbol.name = symbol.full_name;
      }
      symbol.file = fields[1];

      for (int i = 3; i < fields.length; i++)
      {
        string[] splits;

        splits = fields[i].split (":", 0);
        switch (splits[0])
        {
          case "inherits":
            symbol.inherits = splits[1].split (",");
          break;
          default:
            if (symbol.get_class ().find_property (splits[0]) != null)
            {
              symbol.set (splits[0], splits[1]);
            }
          break;
        }
      }
    }
    return symbol;
  }

  /**
   * Execute the ctags command and return stdout.
   */
  private List<Symbol> exec_ctags (string filename) throws Error
  {
    string output;
    string command;
    string[] lines;
    List<Symbol> symbols;

    symbols = new List<Symbol> ();

    /* build ctags command */
    command = "%s --extra=+q --fields=afmiKlnsStz -u -f - \"%s\""
                       .printf (Config.VALIDE_CTAGS_EXEC, filename);
    /* execute command */
    Process.spawn_command_line_sync (command, out output, null, null);

    lines = output.split ("\n", 0);
    for (int i = 0; lines[i] != null; i++)
    {
      if (lines[i] != "")
      {
        Symbol symbol;

        symbol = this.parse_line (lines[i]);
        if (symbol != null)
        {
          symbols.append (symbol);
        }
      }
    }
    return symbols;
  }

  /**
   * Clear the list store and unsensitive the tree view
   */
  private void clear_data ()
  {
    this.model.clear ();
    this.root_nodes = new HashTable<string, Gtk.TreeIter?> (str_hash, str_equal);
    this.box.sensitive = false;
    this.label.set_text ("");
  }

  /**
   * Add a single symbol to the treeview. Parents rows are added as the type
   * whenever they aren't already present.
   */
  private void add_symbol (Symbol symbol)
  {
    if (symbol.kind == "namespace")
    {
      return;
    }

    string markup;
    Gdk.Pixbuf pixbuf;
    Gtk.TreeIter iter;
    Gtk.TreeIter? parent;

    Utils.process_gtk_events ();
    symbol.name = symbol.name.replace ("<", "&lt;");
    symbol.name = symbol.name.replace (">", "&gt;");

    pixbuf = Utils.get_symbol_pixbuf (symbol.kind);

    if (!this.hierarchic_view)
    {
      /* we could do bold here */
      markup = "<b>%s</b>".printf (symbol.kind);

      /* check for existing parent iter */
      parent = this.root_nodes.lookup (markup);
      /* create the parent if it wasn't already found */
      if (parent == null)
      {
        Gtk.TreeIter tmp;

        this.model.append (out tmp, null);
        parent = tmp;
        this.model.set (parent, Col.ICON, pixbuf, Col.NAME, markup);
        this.root_nodes.insert (markup, parent);
      }

      this.model.append (out iter, parent);
      /* add the new symbol */
      this.model.set (iter, Col.ICON, pixbuf,
                            Col.NAME, symbol.name,
                            Col.FILE, symbol.file,
                            Col.LINE, int.parse (symbol.line),
                            Col.LANGUAGE, symbol.language);
    }
    else
    {
      if (symbol.root != null)
      {
        parent = this.root_nodes.lookup (symbol.root.full_name);
        this.model.append (out iter, parent);
      }
      else
      {
        this.model.append (out iter, null);
        parent = iter;
      }
      this.model.set (iter, Col.ICON, pixbuf,
                            Col.NAME, symbol.name,
                            Col.FILE, symbol.file,
                            Col.LINE, int.parse (symbol.line),
                            Col.LANGUAGE, symbol.language);
      this.root_nodes.insert (symbol.full_name, iter);
    }
  }

  private bool populate_symbols_asc ()
  {
    if (AtomicInt.compare_and_exchange (ref this.atom, 0, 1))
    {
      Document document;
      double current_pos;
      Gtk.VScrollbar vcrollbar;

      vcrollbar = this.scrolled_window.get_vscrollbar () as Gtk.VScrollbar;
      current_pos = vcrollbar.get_value();
      this.model = this.tree_view.get_model () as Gtk.TreeStore;
      this.tree_view.set_model (null);
      this.tree_view.freeze_child_notify ();
      this.clear_data ();

      document = this.window.documents.current;
      if (document != null
          && !document.is_new
          && FileUtils.test (document.path, FileTest.EXISTS))
      {
        try
        {
          string parent;
          List<Symbol> symbols;
          HashTable<string, Symbol> root;

          root = new HashTable<string, Symbol> (str_hash, str_equal);
          symbols = this.exec_ctags (document.path);
          foreach (Symbol symbol in symbols)
          {
            symbol.document = document;
            if (symbol.kind != "namespace")
            {
              parent = symbol.full_name;
              Utils.remove_last_element (ref parent, '.');
              symbol.root = root.lookup (parent);
            }
            this.add_symbol (symbol);
            root.insert (symbol.full_name, symbol);
          }
        }
        catch (Error e)
        {
          debug (_("Could not execute ctags: %s\n"), e.message);
        }

        if (this.model.iter_n_children (null) > 0)
        {
          this.box.sensitive = true;
        }
      }
      this.tree_view.set_model (this.model);
      this.tree_view.thaw_child_notify ();
      this.restore_expanded_rows ();

      Utils.process_gtk_events ();
      vcrollbar.set_value(current_pos);
      AtomicInt.set (ref this.atom, 0);
    }
    return false;
  }

  /**
   * Call to re-populate tree view.
   */
  private void populate_symbols ()
  {
    Idle.add (this.populate_symbols_asc);
  }

  private void on_row_activated (Gtk.TreeView sender, Gtk.TreePath path,
                                 Gtk.TreeViewColumn column)
  {
    Gtk.TreeIter iter;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get_iter (out iter, path);

    /* expand/collapse row if it has children (and return) */
    if (!this.hierarchic_view && model.iter_has_child (iter))
    {
      if (this.tree_view.is_row_expanded (path))
      {
        this.tree_view.collapse_row (path);
      }
      else
      {
        this.tree_view.expand_row (path, false);
      }
    }
    else
    {
      /* otherwise, show the file name and line number and flash statusbar */
      int line;
      string name;
      string filename;

      model.get (iter, Col.NAME, out name,  Col.LINE, out line,
                 Col.FILE, out filename);
      try
      {
        Document document;

        document = this.window.documents.create (filename);
        document.text_view.goto_line (line - 1);
        document.grab_focus ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  private void on_cursor_changed (Gtk.TreeView sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreeModel model;
    Gtk.TreeSelection selection;

    model = this.tree_view.get_model ();
    selection = this.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      int line;
      string name;
      string filename;

      model.get (iter, Col.NAME, out name, Col.FILE, out filename,
                       Col.LINE, out line);

      if (filename != null)
      {
        string label_text;
        string status_message;

        label_text = "%s Line:%d".printf (Path.get_basename (filename), line);
        this.label.set_text (label_text);

        status_message = "Double-click to jump to '%s'".printf (name);
        this.window.statusbar.flash_message (this.status_context, status_message);
      }
    }
  }

  private void on_row_expanded (Gtk.TreeView sender, Gtk.TreeIter iter,
                                Gtk.TreePath path)
  {
    string name;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get (iter, Col.NAME, out name);
    return_if_fail (name != null);

    this.add_expanded_row (name);
  }

  private void on_row_collapsed (Gtk.TreeView sender, Gtk.TreeIter iter,
                                 Gtk.TreePath path)
  {
    string name;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get (iter, Col.NAME, out name);
    return_if_fail (name != null);

    this.remove_expanded_row (name);
  }

  public SymbolBrowser (string path)
  {
    Object (path: path);
  }

  construct
  {
    Gtk.TreeStore store;
    Gtk.TreeViewColumn col;
    Gtk.CellRendererText renderer;
    Gtk.CellRendererPixbuf renderer_pixbuf;

    AtomicInt.set (ref this.atom, 0);

    this.box = new Gtk.VBox (false, 0);

    /* create some nice padding */
    this.box.set_spacing (2);
    this.box.set_border_width (2);

    /* create and pack tree view */
    this.scrolled_window = new Gtk.ScrolledWindow (null, null);
    this.scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    this.scrolled_window.set_shadow_type (Gtk.ShadowType.IN);
    this.box.pack_start (this.scrolled_window, true, true, 0);

    this.tree_view = new Gtk.TreeView ();
    this.tree_view.set_rules_hint (false);
    this.tree_view.set_headers_visible (true);
    this.tree_view.set_headers_clickable (true);
    this.scrolled_window.add (this.tree_view);

    /* create and pack label */
    this.label = new Gtk.Label ("");
    this.label.set_alignment ((float)0.0, (float)0.5);
    this.box.pack_start (this.label, false, true, 0);

    /* create model */
    store = new Gtk.TreeStore (Col.NB, typeof (Gdk.Pixbuf), typeof (string),
                               typeof (string), typeof (int), typeof (string),
                               typeof (Object));
    store.set_sort_column_id (Col.NAME, Gtk.SortType.ASCENDING);

    /* main column */
    col = new Gtk.TreeViewColumn ();
    col.set_title(_("Symbol"));

    renderer_pixbuf = new Gtk.CellRendererPixbuf ();
    col.pack_start (renderer_pixbuf, false);
    col.set_attributes (renderer_pixbuf, "pixbuf", Col.ICON);

    renderer = new Gtk.CellRendererText ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "markup", Col.NAME);
    this.tree_view.append_column (col);

    /* additional columns */

    this.language_column = new Gtk.TreeViewColumn ();
    this.show_language_column = this.window.config_manager.get_boolean ("SymbolBrowser", "show-language");
    this.language_column.set_title (_("Language"));
    renderer = new Gtk.CellRendererText ();
    this.language_column.pack_start (renderer, true);
    this.language_column.set_attributes (renderer, "text", Col.LANGUAGE);
    this.tree_view.append_column (this.language_column);

    this.source_file_column = new Gtk.TreeViewColumn ();
    this.show_source_file_column = this.window.config_manager.get_boolean ("SymbolBrowser", "show-source-file");
    this.source_file_column.set_title (_("File"));
    renderer = new Gtk.CellRendererText ();
    this.source_file_column.pack_start (renderer, true);
    this.source_file_column.set_attributes (renderer, "text", Col.FILE);
    this.tree_view.append_column (this.source_file_column);

    this.line_number_column = new Gtk.TreeViewColumn ();
    this.show_line_number_column = this.window.config_manager.get_boolean ("SymbolBrowser", "show-line-number");
    this.line_number_column.set_title (_("Line"));
    renderer = new Gtk.CellRendererText ();
    this.line_number_column.pack_start (renderer, true);
    this.line_number_column.set_attributes (renderer, "text", Col.LINE);
    this.tree_view.append_column (this.line_number_column);

    this.tree_view.set_model (store);

    /* attach signal callbacks for tree view */
    this.tree_view.row_activated.connect (this.on_row_activated);
    this.tree_view.cursor_changed.connect (this.on_cursor_changed);
    this.tree_view.row_expanded.connect (this.on_row_expanded);
    this.tree_view.row_collapsed.connect (this.on_row_collapsed);

    this.status_context = this.window.statusbar.get_context_id ("valide_symbol_browser");

    this.box.sensitive = false;

    this.window.documents.tab_removed.connect (this.populate_symbols);
    this.window.documents.tab_changed.connect (this.populate_symbols);
    this.window.documents.tab_saved.connect (this.populate_symbols);

    try
    {
      Utils.register_icon (Path.build_filename (Config.PIXMAPS_DIR, "plugins",
                                                "symbol-browser.png"),
                           "symbol-browser-plugin-icon");
    }
    catch (Error e)
    {
      debug (e.message);
    }
    this.window.add_widget (this.box, "symbol-browser-plugin", _("Symbols"),
                            Window.Placement.LEFT, "symbol-browser-plugin-icon");
  }

  ~SymbolBrowser ()
  {
    this.window.documents.tab_removed.disconnect (this.populate_symbols);
    this.window.documents.tab_changed.disconnect (this.populate_symbols);
    this.window.documents.tab_saved.disconnect (this.populate_symbols);

    this.window.remove_widget (this.box);
  }

  public Gtk.Widget create_configure_dialog ()
  {
    Frame frame;
    Gtk.VBox vbox;
    Gtk.CheckButton check;

    frame = new Frame (_("Additionnal columns"));

    vbox = new Gtk.VBox (true, 0);
    frame.add (vbox);

    check = new Gtk.CheckButton.with_label (_("Programming language"));
    check.set_active (this.show_language_column);
    check.toggled.connect ((s) => {
      this.show_language_column = s.get_active ();
    });
    vbox.pack_start (check, true, true, 0);

    check = new Gtk.CheckButton.with_label (_("Source file"));
    check.set_active (this.show_source_file_column);
    check.toggled.connect ((s) => {
      this.show_source_file_column = s.get_active ();
    });
    vbox.pack_start (check, true, true, 0);

    check = new Gtk.CheckButton.with_label (_("Line number"));
    check.set_active (this.show_line_number_column);
    check.toggled.connect ((s) => {
      this.show_line_number_column = s.get_active ();
    });
    vbox.pack_start (check, true, true, 0);

    check = new Gtk.CheckButton.with_label (_("Hierarchic view"));
    check.set_active (this.hierarchic_view);
    check.toggled.connect ((s) => {
      this.hierarchic_view = s.get_active ();
    });
    vbox.pack_start (check, true, true, 0);

    return frame;
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (SymbolBrowser);
}

public Gtk.Widget create_configure_dialog (SymbolBrowser self)
{
  return self.create_configure_dialog ();
}

