/* start-page.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

private struct Item
{
  public string uri;
  public string text;
  public Gdk.Pixbuf icon;
  public Callback callback;
}

/**
 * The start page widget
 */
public class StartPage : Plugin, Object
{
  private Gtk.Table table;
  private Gdk.Pixbuf logo;
  private Gtk.EventBox event_box;
  private Gtk.Widget file_frame;
  private Gtk.Widget project_frame;
  private uint status_context = -1;
  private Gtk.ScrolledWindow scrolled_window;

  /**
   * @see Valide.Plugin.window
   */
  public Window window { get; construct set; }

  /**
   * @see Valide.Plugin.path
   */
  public string path { get; construct set; }

  [CCode (instance_pos = -1)]
  private void create_project (Gtk.Button sender)
  {
    this.window.projects.create ();
  }

  [CCode (instance_pos = -1)]
  private void open_project (Gtk.Button sender)
  {
    string? uri;

    uri = sender.get_data ("uri");
    try
    {
      this.window.projects.open (uri);
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  [CCode (instance_pos = -1)]
  private void create_file (Gtk.Button sender)
  {
    try
    {
      this.window.documents.create ();
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  [CCode (instance_pos = -1)]
  private void open_file (Gtk.Button sender)
  {
    string? uri;

    uri = sender.get_data ("uri");
    try
    {
      if (uri == null)
      {
        this.window.documents.open ();
      }
      else
      {
        this.window.documents.create (uri);
      }
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  [CCode (instance_pos = -1)]
  private void open_link (Gtk.Button sender)
  {
    string? uri;

    uri = sender.get_data ("uri");
    if (uri != null)
    {
      try
      {
        if (Config.OS == "win32")
             Process.spawn_command_line_async ("cmd /c start /max " + uri);
        else
             AppInfo.launch_default_for_uri (uri, null);
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  private void send_message (string message)
  {
    if (this.status_context == -1)
    {
      this.status_context = this.window.statusbar.get_context_id ("valide_start_page");
    }
    this.window.statusbar.push (this.status_context, message);
  }

  private bool button_leave_notify_event (Gtk.Widget sender,
                                          Gdk.EventCrossing event)
  {
    this.send_message ("");
    return false;
  }

  private bool button_enter_notify_event (Gtk.Widget sender,
                                          Gdk.EventCrossing event)
  {
    string? uri;
    Gtk.Button button;
    string status_message;

    button = sender as Gtk.Button;
    uri = button.get_data ("uri");
    if (uri != null)
    {
      uri = Utils.replace_home_dir_with_tilde (uri);
      status_message = _("Open %s").printf (uri);
    }
    else
    {
      status_message = "";
    }
    this.send_message (status_message);
    return false;
  }

  private Gtk.Button create_button (string? uri, string text, Gdk.Pixbuf? icon)
  {
    string markup;
    Gtk.HBox hbox;
    Gtk.Label label;
    Gtk.Button button;

    button = new Gtk.Button ();
    button.set_relief (Gtk.ReliefStyle.NONE);
    button.set_alignment (0, 0.5f);

    hbox = new Gtk.HBox (false, 5);
    button.add (hbox);

    if (icon != null)
    {
      Gtk.Image image;

      image = new Gtk.Image.from_pixbuf (icon);
      hbox.pack_start (image, false, false, 0);
    }

    markup = Markup.printf_escaped ("<span underline=\"single\" foreground=\"#5a7ac7\">%s</span>", text);
    label = new Gtk.Label (markup);
    label.set_use_markup (true);
    hbox.pack_start (label, false, false, 0);

    button.set_data ("uri", uri);
    return button;
  }

  private Gtk.Widget create_frame (string title, List<Item?> items)
  {
    Frame frame;
    Gtk.VBox vbox;
    Gtk.Label label;

    frame = new Frame (title);

    vbox = new Gtk.VBox (false, 5);
    vbox.set_border_width (5);
    frame.add (vbox);

    if (items != null)
    {
      foreach (Item item in items)
      {
        Gtk.Button button;

        button = this.create_button (item.uri, item.text, item.icon);
        button.enter_notify_event.connect (this.button_enter_notify_event);
        button.leave_notify_event.connect (this.button_leave_notify_event);
  
        Signal.connect (button, "clicked", item.callback, this);
        vbox.pack_start (button, false, false, 0);
      }
    }
    else
    {
      label = new Gtk.Label (_("No item"));
      vbox.pack_start (label, false, false, 0);
    }
    return frame;
  }

  private Gtk.Widget populate_frame (List<Gtk.RecentInfo> infos, string title,
                                     Callback callback)
  {
    Item item;
    List<Item?> items;

    items = new List<Item?> ();
    foreach (Gtk.RecentInfo info in infos)
    {
      item = Item ();
      item.icon = null;
      item.uri = info.get_uri ();
      item.text = Path.get_basename (item.uri);
      item.callback = callback;

      items.append (item);
    }
    return this.create_frame (title, items);
  }

  private void populate_file_frame ()
  {
    if (this.file_frame != null)
    {
      this.file_frame.destroy ();
    }
    if (this.window.recent_manager.max_recent_files > 0)
    {
      this.file_frame = this.populate_frame (this.window.recent_manager.recent_files,
                                             _("Recent files"),
                                             (Callback)this.open_file);
      this.table.attach (this.file_frame, 1, 2, 1, 2, Gtk.AttachOptions.FILL,
                         Gtk.AttachOptions.FILL, 0, 0);
      this.file_frame.show_all ();
    }
  }

  private void populate_project_frame ()
  {
    if (this.project_frame != null)
    {
      this.project_frame.destroy ();
    }
    if (this.window.recent_manager.max_recent_projects > 0)
    {
      this.project_frame = this.populate_frame (this.window.recent_manager.recent_projects,
                                                _("Recent projects"),
                                                (Callback)this.open_project);
      this.table.attach (this.project_frame, 0, 1, 1, 2, Gtk.AttachOptions.FILL,
                         Gtk.AttachOptions.FILL, 0, 0);
      this.project_frame.show_all ();
    }
  }

  private bool on_expose_event_cb (Gtk.Widget widget, Gdk.EventExpose event)
  {
    Cairo.Context cr;
    Cairo.Pattern pattern;

    cr = Gdk.cairo_create (widget.window);

    pattern = new Cairo.Pattern.linear (0, 0, 0, widget.allocation.height);

    if (widget.get_screen ().get_rgba_colormap () != null &&
        widget.is_composited ())
    {
      cr.set_source_rgba (1.0, 1.0, 1.0, 0.0); /* transparent */
    }
    else
    {
      cr.set_source_rgb (1.0, 1.0, 1.0); /* opaque white */
    }

    cr.set_operator (Cairo.Operator.SOURCE);
    cr.paint ();

    pattern.add_color_stop_rgba (0.0, 0.2, 0.6, 0.2, 1.0);
    pattern.add_color_stop_rgba (0.18, 1.0, 1.0, 1.0, 1.0);

    cr.set_source (pattern);

    cr.set_operator (Cairo.Operator.OVER);

    cr.paint ();

    cr = Gdk.cairo_create (widget.window);

    Gdk.cairo_set_source_pixbuf (cr, this.logo, 20, 20);

    cr.paint ();

    if (widget is Gtk.Container)
    {
      Gtk.Container container;

      container = widget as Gtk.Container;
      foreach (Gtk.Widget child in container.get_children ())
      {
        container.propagate_expose (child, event);
      }
    }
    return true;
  }

  construct
  {
    Gtk.VBox vbox;
    Gtk.Label label;
    string copyright;
    Gtk.Widget frame;
    List<Item?> items;
    uint logo_offset = 20;
    Gtk.Alignment alignment;

    this.scrolled_window = new Gtk.ScrolledWindow (null, null);
    this.scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC,
                                     Gtk.PolicyType.AUTOMATIC);

    this.event_box = new Gtk.EventBox ();
    this.scrolled_window.add_with_viewport (this.event_box);

    try
    {
      this.logo = new Gdk.Pixbuf.from_file (Path.build_filename (this.path, "logo.png"));
      this.event_box.expose_event.connect (this.on_expose_event_cb);
    }
    catch (Error e)
    {
      warning (e.message);
    }

    alignment = new Gtk.Alignment (0.5f, 0.5f, 1.0f, 1.0f);
    alignment.set_padding (logo_offset + logo.get_height () + logo_offset, 0,
                           logo_offset, logo_offset);
    this.event_box.add (alignment);

    vbox = new Gtk.VBox (false, 5);
    alignment.add (vbox);

    table = new Gtk.Table (2, 2, true);
    vbox.pack_start (table, false, false, 0);

    items = new List<Item?> ();
    /* New project */
    Item item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.NEW, Gtk.IconSize.BUTTON);
    item.uri = null;
    item.text = _("New project");
    item.callback = (Callback)this.create_project;
    items.append (item);

    /* Open project */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.OPEN, Gtk.IconSize.BUTTON);
    item.uri = null;
    item.text = _("Open project");
    item.callback = (Callback)this.open_project;
    items.append (item);

    /* New file */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.NEW, Gtk.IconSize.BUTTON);
    item.uri = null;
    item.text = _("New file");
    item.callback = (Callback)this.create_file;
    items.append (item);

    /* Open file */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.OPEN, Gtk.IconSize.BUTTON);
    item.uri = null;
    item.text = _("Open file");
    item.callback = (Callback)this.open_file;
    items.append (item);

    frame = this.create_frame (_("Common actions"), items);
    this.table.attach (frame, 0, 1, 0, 1, Gtk.AttachOptions.FILL,
                       Gtk.AttachOptions.FILL, 0, 0);

    items = new List<Item?> ();
    /* Website */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.HOME, Gtk.IconSize.BUTTON);
    item.uri = Config.WEBSITE;
    item.text = _("Website");
    item.callback = (Callback)this.open_link;
    items.append (item);

    /* Report a bug */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.CANCEL, Gtk.IconSize.BUTTON);
    item.uri = "https://bugs.launchpad.net/valide";
    item.text = _("Report a bug");
    item.callback = (Callback)this.open_link;
    items.append (item);

    /* Visit the user group */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.HELP, Gtk.IconSize.BUTTON);
    item.uri = "https://answers.launchpad.net/valide";
    item.text = _("Visit the user group");
    item.callback = (Callback)this.open_link;
    items.append (item);

    /* Online API documentation */
    item = Item ();
    item.icon = Utils.get_pixbuf_for_stock (Gtk.Stock.INFO, Gtk.IconSize.BUTTON);
    item.uri = "http://www.valadoc.org/";
    item.text = _("Online API documentation");
    item.callback = (Callback)this.open_link;
    items.append (item);

    frame = this.create_frame (_("Links"), items);
    this.table.attach (frame, 1, 2, 0, 1, Gtk.AttachOptions.FILL,
                       Gtk.AttachOptions.FILL, 0, 0);

    this.populate_project_frame ();
    this.populate_file_frame ();

    this.window.recent_manager.update_recent_projects.connect (this.populate_project_frame);
    this.window.recent_manager.update_recent_files.connect (this.populate_file_frame);

    copyright = Config.COPYRIGHT.replace ("The Val(a)IDE team",
                                          "<a href=\"https://launchpad.net/~valide\">The Val(a)IDE team</a>");
    label = new Gtk.Label (copyright);
    label.use_markup = true;
    vbox.pack_end (label, false, true, 5);

    vbox.pack_end (new Gtk.HSeparator (), false, true, 0);

    this.window.add_widget (this.scrolled_window, "start-page-plugin",
                            _("Start page"), Window.Placement.CENTER,
                            Gtk.Stock.INFO);

    this.window.show.connect (() => {
      if (this.window.documents.current == null
          || this.window.documents.current.is_new)
      {
        this.window.present_widget (this.scrolled_window);
      }
    });
  }

  ~StartPagePlugin ()
  {
    this.window.recent_manager.update_recent_projects.disconnect (this.populate_project_frame);
    this.window.recent_manager.update_recent_files.disconnect (this.populate_file_frame);
    this.window.remove_widget (this.scrolled_window);
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (StartPage);
}

