/* opened-documents.vala
 *
 * Copyright (C) 2009-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

public class OpenedDocuments : Plugin, Object
{
  /* enum for tree model columns, not all is currently used */
  enum Col
  {
    ICON,
    NAME,
    DOCUMENT,
    STYLE,
    TOOLTIP,
    NB
  }

  private Gtk.VBox box;
  private uint ui_id = -1;
  private bool on_idle = false;
  private Gtk.TreeView tree_view;
  private DocumentManager documents;
  private Gtk.ActionGroup action_group;

  /**
   * @see Plugin.path
   */
  public string path { get; construct set; }

  /**
   * @see Plugin.window
   */
  public Window window { get; construct set; }

  private void on_action_activate (Gtk.Action action)
  {
    string[] tokens;

    tokens = action.name.split ("-");
    if (tokens != null)
    {
      int page_num;

      page_num = int.parse (tokens[2]);
      this.window.documents.set_current_page (page_num - 1);
    }
    else
    {
      debug ("invalid action name: %s", action.name);
    }
  }

  private void populate_menu ()
  {
    int i = 0;
    int active;
    UIManager ui_manager;
    unowned SList<Gtk.RadioAction> group = null;

    ui_manager = this.window.ui_manager;
    if (this.ui_id != -1)
    {
      ui_manager.remove_ui (this.ui_id);
    }
    this.ui_id = ui_manager.new_merge_id ();

    this.action_group = new Gtk.ActionGroup ("document-opened");
    ui_manager.insert_action_group (this.action_group, 0);

    active = this.documents.get_current_page () + 1;
    foreach (Document document in this.documents.documents)
    {
      string accel;
      string action_name;
      Gtk.RadioAction action;

      i++;

      /* alt + 1, 2, 3... 0 to switch to the first ten tabs */
      accel = (i < 11) ? "<alt>%d".printf (i % 10) : null;

      action_name = "document-opened-%d".printf (i);
		  action = new Gtk.RadioAction (action_name, document.filename,
					                          "Activate %s".printf (document.path),
					                          null, i);
      if (i == active)
      {
        action.set_active (true);
      }
      if (group != null)
      {
        action.set_group (group);
      }
      /* note that group changes each time we add an action, so it must be updated */
		  group = action.get_group ();

      this.action_group.add_action_with_accel (action, accel);

      action.changed.connect (this.on_action_activate);

      ui_manager.add_ui (this.ui_id, "/menubar/document/document-ops-1",
                         action_name, action_name,
                         Gtk.UIManagerItemType.MENUITEM, false);
    }
  }

  private void populate_treeview ()
  {
    Gtk.TreeIter iter;
    Gtk.ListStore model;

    model = this.tree_view.get_model () as Gtk.ListStore;
    model.clear ();

    foreach (Document document in this.documents.documents)
    {
      string tooltip;
      Pango.Style style;

      if (document.is_save)
      {
        style = Pango.Style.NORMAL;
      }
      else
      {
        style = Pango.Style.ITALIC;
      }
      tooltip = document.get_tooltips ();
      model.append (out iter);
      model.set (iter, Col.ICON, document.get_icon (),
                 Col.NAME, document.filename,
                 Col.DOCUMENT, document,
                 Col.STYLE, style,
                 Col.TOOLTIP, tooltip);
      if (document == this.documents.current)
      {
        this.tree_view.get_selection ().select_iter (iter);
      }
    }
  }

  private bool populate_asc ()
  {
    if (!this.on_idle)
    {
      this.on_idle = true;
      this.populate_menu ();
      this.populate_treeview ();
      this.on_idle = false;
    }
    return false;
  }

  private void populate ()
  {
    Idle.add (this.populate_asc);
  }

  private void on_cursor_change (Gtk.TreeView sender)
  {
    Document document;
    Gtk.TreeIter iter;
    Gtk.TreeModel model;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();

    if (selection.get_selected (out model, out iter))
    {
      model.get (iter, Col.DOCUMENT, out document);
      this.documents.current = document;
    }
  }

  construct
  {
    Gtk.ListStore store;
    Gdl.DockItem parent;
    Gtk.TreeViewColumn col;
    Gtk.CellRenderer renderer;
    Gtk.ScrolledWindow scrolled;

    this.box = new Gtk.VBox (false, 0);

    /* create some nice padding */
    this.box.set_spacing (2);
    this.box.set_border_width (2);

    /* create and pack tree view */
    scrolled = new Gtk.ScrolledWindow (null, null);
    scrolled.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    scrolled.set_shadow_type (Gtk.ShadowType.IN);
    this.box.pack_start (scrolled, true, true, 0);

    this.tree_view = new Gtk.TreeView ();
    this.tree_view.set_rules_hint (false);
    this.tree_view.set_headers_visible (false);
    scrolled.add (this.tree_view);

    /* create model */
    store = new Gtk.ListStore (Col.NB, typeof (Gdk.Pixbuf), typeof (string),
                               typeof (Document), typeof (Pango.Style),
                               typeof (string));
    store.set_sort_column_id (Col.NAME, Gtk.SortType.ASCENDING);

    /* main column */
    col = new Gtk.TreeViewColumn ();
    col.set_title(_("File"));

    renderer = new Gtk.CellRendererPixbuf ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "pixbuf", Col.ICON);

    renderer = new Gtk.CellRendererText ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "markup", Col.NAME, "style", Col.STYLE);

    this.tree_view.append_column (col);
    this.tree_view.set_model (store);

    this.tree_view.set_tooltip_column (Col.TOOLTIP);

    /* attach signal callbacks for tree view */
    this.tree_view.cursor_changed.connect (this.on_cursor_change);

    this.documents = this.window.documents;

    /* Sync call to prevent a race condition */
    this.populate_asc ();

    this.documents.tab_removed.connect (this.populate);
    this.documents.tab_added.connect (this.populate);
    this.documents.tab_state_changed.connect (this.populate);
    this.documents.tab_changed.connect (this.populate);
    this.documents.page_reordered.connect (this.populate);

    this.window.add_widget (this.box, "opened-documents-plugin", _("Documents"),
                            Window.Placement.LEFT, Gtk.Stock.FILE);
  }

  ~OpenedDocuments ()
  {
    this.documents.tab_removed.disconnect (this.populate);
    this.documents.tab_added.disconnect (this.populate);
    this.documents.tab_state_changed.disconnect (this.populate);
    this.documents.tab_changed.disconnect (this.populate);
    this.documents.page_reordered.disconnect (this.populate);

    if (this.ui_id > 0)
    {
      this.window.ui_manager.remove_action_group (this.action_group);
    }
    this.window.remove_widget (this.box);
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (OpenedDocuments);
}

