#!/bin/bash
#
# inspired by the clutter project
#
# This script will download and setup a cross compilation environment
# for targetting Win32 from Linux. It can also be used to build on
# Windows under the MSYS/MinGW environment.

GTK_URL="http://ftp.gnome.org/pub/gnome/binaries/win32";
GTK_BINARIES=( \
  glib/2.22/glib{-dev,}_2.22.4-1_win32.zip \
  gtk+/2.18/gtk+{-dev,}_2.18.5-1_win32.zip \
  pango/1.26/pango{-dev,}_1.26.1-1_win32.zip \
  atk/1.28/atk{-dev,}_1.28.0-1_win32.zip \
  gtksourceview/2.10/gtksourceview{-dev,}-2.10.0.zip \
  dependencies/cairo{-dev,}_1.8.8-3_win32.zip \
  dependencies/gettext-runtime-{dev-,}0.17-1.zip \
  dependencies/fontconfig{-dev,}_2.8.0-1_win32.zip \
  dependencies/freetype{-dev,}_2.3.11-1_win32.zip \
  dependencies/libxml2{-dev,}_2.7.7-1_win32.zip \
  dependencies/expat_2.0.1-1_win32.zip \
  dependencies/libiconv-1.9.1.bin.woe32.zip \
  dependencies/libpng_1.2.40-1_win32.zip \
  dependencies/jpeg_7-1_win32.zip \
  dependencies/libtiff_3.9.1-1_win32.zip \
  dependencies/zlib-1.2.3.zip
);

MSYS_URL="http://downloads.sourceforge.net/project/mingw/MSYS%20w32api";
MSYS_BINARIES=(\
  w32api-3.14-3/w32api-3.14-3-msys-1.0.12-dev.tar.lzma
);

VALA_URL="http://valide.googlecode.com/files";
VALA_BINARIES=(\
  vala-0.8.1.zip
);

GTK_SRC_URL="http://ftp.gnome.org/pub/gnome/sources/";
GTK_SRC=(\
  gdl/2.30/gdl-2.30.0.tar.gz \
);

VALIDE_SVN="http://valide.googlecode.com/svn/tags/0.8"

function download_file ()
{
  local url="$1"; shift;
  local filename="$1"; shift;

  case "$DOWNLOAD_PROG" in
    curl)
      curl -C - -o "$DOWNLOAD_DIR/$filename" "$url";
    ;;
    *)
      $DOWNLOAD_PROG -O "$DOWNLOAD_DIR/$filename" -c "$url";
    ;;
  esac;

  if [ $? -ne 0 ]; then
    echo "Downloading ${url} failed.";
    exit 1;
  fi;
}

function guess_dir ()
{
  local var="$1"; shift;
  local suffix="$1"; shift;
  local msg="$1"; shift;
  local prompt="$1"; shift;
  local dir="${!var}";

  if [ -z "$dir" ]; then
    echo "Please enter ${msg}.";
    dir="$PWD/$suffix";
    read -r -p "$prompt [$dir] ";
    if [ -n "$REPLY" ]; then
      dir="$REPLY";
    fi;
  fi;

  eval $var="\"$dir\"";

  if [ ! -d "$dir" ]; then
    if ! mkdir -p "$dir"; then
      echo "Error making directory $dir";
      exit 1;
    fi;
  fi;
}

function y_or_n ()
{
  local prompt="$1"; shift;

  while true; do
    read -p "${prompt} [y/n] " -n 1;
    echo;
    case "$REPLY" in
      y) return 0 ;;
      n) return 1 ;;
      *) echo "Please press y or n" ;;
    esac;
  done;
}

function do_unlzma ()
{
  do_unlzma_d "$ROOT_DIR" "$@";
}

function do_unlzma_d ()
{
  local exdir="$1"; shift;
  local zipfile="$1"; shift;

  echo "cat $zipfile | unlzma -c | tar -C $exdir xvf -"
  cat "$zipfile" | unlzma -c | tar -C "$exdir" -xvf -

  if [ "$?" -ne 0 ]; then
    echo "Failed to extract $zipfile";
    exit 1;
  fi;
}

function do_unzip ()
{
  do_unzip_d "$ROOT_DIR" "$@";
}

function do_unzip_d ()
{
  local exdir="$1"; shift;
  local zipfile="$1"; shift;

  unzip -o -q -d "$exdir" "$zipfile" "$@";

  if [ "$?" -ne 0 ]; then
    echo "Failed to extract $zipfile";
    exit 1;
  fi;
}

function find_compiler ()
{
  local gccbin fullpath;

  if [ -z "$MINGW_TOOL_PREFIX" ]; then
    for gccbin in i{3,4,5,6}86-mingw32{,msvc}-gcc; do
      fullpath="`which $gccbin 2>/dev/null`";
      if [ "$?" -eq 0 ]; then
        MINGW_TOOL_PREFIX="${fullpath%%gcc}";
        break;
      fi;
    done;
    if [ -z "$MINGW_TOOL_PREFIX" ]; then
      echo;
      echo "No suitable cross compiler was found.";
      echo;
      echo "If you already have a compiler installed,";
      echo "please set the MINGW_TOOL_PREFIX variable";
      echo "to point to its location without the";
      echo "gcc suffix (eg: \"/usr/bin/i386-mingw32-\").";
      echo;
      echo "If you are using Ubuntu, you can install a";
      echo "compiler by typing:";
      echo;
      echo " sudo apt-get install mingw32";
      echo;
      echo "Otherwise you can try following the instructions here:";
      echo;
      echo " http://www.libsdl.org/extras/win32/cross/README.txt";

      exit 1;
    fi;
  fi;

  export ADDR2LINE="${MINGW_TOOL_PREFIX}addr2line"
  export AS="${MINGW_TOOL_PREFIX}as"
  export CC="${MINGW_TOOL_PREFIX}gcc"
  export CPP="${MINGW_TOOL_PREFIX}cpp"
  export CPPFILT="${MINGW_TOOL_PREFIX}c++filt"
  export CXX="${MINGW_TOOL_PREFIX}g++"
  export DLLTOOL="${MINGW_TOOL_PREFIX}dlltool"
  export DLLWRAP="${MINGW_TOOL_PREFIX}dllwrap"
  export GCOV="${MINGW_TOOL_PREFIX}gcov"
  export LD="${MINGW_TOOL_PREFIX}ld"
  export NM="${MINGW_TOOL_PREFIX}nm"
  export OBJCOPY="${MINGW_TOOL_PREFIX}objcopy"
  export OBJDUMP="${MINGW_TOOL_PREFIX}objdump"
  export READELF="${MINGW_TOOL_PREFIX}readelf"
  export SIZE="${MINGW_TOOL_PREFIX}size"
  export STRINGS="${MINGW_TOOL_PREFIX}strings"
  export WINDRES="${MINGW_TOOL_PREFIX}windres"
  export AR="${MINGW_TOOL_PREFIX}ar"
  export RANLIB="${MINGW_TOOL_PREFIX}ranlib"
  export STRIP="${MINGW_TOOL_PREFIX}strip"

  TARGET="${MINGW_TOOL_PREFIX##*/}";
  TARGET="${TARGET%%-}";

  echo "Using compiler $CC and target $TARGET";
}

# If a download directory hasn't been specified then try to guess one
# but ask for confirmation first
guess_dir DOWNLOAD_DIR "downloads" \
  "the directory to download to" "Download directory";

# Try to guess a download program if none has been specified
if [ -z "$DOWNLOAD_PROG" ]; then
  # If no download program has been specified then check if wget or
  # curl exists
  #wget first, because my curl can't download libsdl...
  for x in wget curl; do
    if [ "`type -t $x`" != "" ]; then
      DOWNLOAD_PROG="$x";
      break;
    fi;
  done;

  if [ -z "$DOWNLOAD_PROG" ]; then
    echo "No DOWNLOAD_PROG was set and neither wget nor curl is ";
    echo "available.";
    exit 1;
  fi;
fi;

# If a download directory hasn't been specified then try to guess one
# but ask for confirmation first
guess_dir ROOT_DIR "valide-cross" \
    "the root prefix for the build environment" "Root dir";
SLASH_SCRIPT='s/\//\\\//g';
quoted_root_dir=`echo "$ROOT_DIR" | sed "$SLASH_SCRIPT" `;

##
# Download files
##

for bin in "${GTK_BINARIES[@]}"; do
  bn="${bin##*/}";
  download_file "$GTK_URL/$bin" "$bn"
done;

for bin in "${MSYS_BINARIES[@]}"; do
  bn="${bin##*/}";
  download_file "$MSYS_URL/$bin" "$bn"
done;

for bin in "${VALA_BINARIES[@]}"; do
  bn="${bin##*/}";
  download_file "$VALA_URL/$bin" "$bn"
done;

for src in "${GTK_SRC[@]}"; do
  sc="${src##*/}";
  download_file "$GTK_SRC_URL/$src" "$sc"
done;

##
# Extract files
##

for bin in "${GTK_BINARIES[@]}"; do
  echo "Extracting $bin...";
  bn="${bin##*/}";
  do_unzip "$DOWNLOAD_DIR/$bn";
done;

for bin in "${MSYS_BINARIES[@]}"; do
  echo "Extracting $bin...";
  bn="${bin##*/}";
  do_unlzma "$DOWNLOAD_DIR/$bn";
done;

for bin in "${VALA_BINARIES[@]}"; do
  echo "Extracting $bin...";
  bn="${bin##*/}";
  do_unzip "$DOWNLOAD_DIR/$bn";
done;

echo "Fixing pkgconfig files...";
for x in "$ROOT_DIR/lib/pkgconfig/"*.pc; do
    sed "s/^prefix=.*\$/prefix=${quoted_root_dir}/" \
	< "$x" > "$x.tmp";
    mv "$x.tmp" "$x";
done;

mv "$ROOT_DIR/README.libiconv" "$ROOT_DIR/share/doc"

##
# Build
##

export PKG_CONFIG_PATH="$ROOT_DIR/lib/pkgconfig:$PKG_CONFIG_PATH";

export LDFLAGS="-L$ROOT_DIR/lib -mno-cygwin $LDFLAGS"
export CPPFLAGS="-I$ROOT_DIR/include -I$ROOT_DIR/include/w32api $CPPFLAGS"
export CFLAGS="-I$ROOT_DIR/include -mno-cygwin -mms-bitfields -march=i686 ${CFLAGS:-"-g"}"
export CXXFLAGS="-I$ROOT_DIR/include -mno-cygwin -mms-bitfields -march=i686 ${CFLAGS:-"-g"}"

find_compiler;

# Dependancies
for src in "${GTK_SRC[@]}"; do
  sc="${src##*/}";

  echo "building ${sc%.tar.gz}...";

  tar -zxf "$DOWNLOAD_DIR/$sc";
  if [ "$?" -ne 0 ]; then
    echo "extract failed";
    exit 1;
  fi;

  SRC_BUILD_DIR="$PWD/${sc%.tar.gz}";
  ( cd "$SRC_BUILD_DIR" && ./configure --prefix="$ROOT_DIR" \
    --host="$TARGET" --target="$TARGET");
  if [ "$?" -ne 0 ]; then
    echo "configure failed";
    exit 1;
  fi;
  ( cd "$SRC_BUILD_DIR" && make all install );
  if [ "$?" -ne 0 ]; then
    echo "make failed";
    exit 1;
  fi;
done;

# Code
if y_or_n "Do you want to checkout and build valide?"; then

  guess_dir VALIDE_BUILD_DIR "valide" \
   "the build directory for valide" "Build dir";
  svn export --force $VALIDE_SVN $VALIDE_BUILD_DIR;
  if [ "$?" -ne 0 ]; then
    echo "checkout failed";
    exit 1;
  fi;

  ( cd "$VALIDE_BUILD_DIR" && ./waf configure --prefix="$ROOT_DIR" \
    --with-target-platform="Windows" --without-libunique);
  if [ "$?" -ne 0 ]; then
    echo "configure failed";
    exit 1;
  fi;
  ( cd "$VALIDE_BUILD_DIR" && ./waf build install );
  if [ "$?" -ne 0 ]; then
    echo "build failed";
    exit 1;
  fi;
fi;

