/* utils.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * A class for represente a rectangular position
 */
public struct Valide.Position
{
  public int start_line;
  public int end_line;
  public int start_row;
  public int end_row;
}

/**
 * Miscellaneous functions
 */
public class Valide.Utils
{
  private static HashTable<string, Gdk.Pixbuf> icons = null;

  public class Color
  {
    public const string INFO = "#0000FF";
    public const string SUCCESS = "#19A630";
    public const string WARNING = "#ED7F10";
    public const string ERROR = "#FF0000";
    public const string DEFAULT = "#000000";
  }

  /**
   * Gets the path for filename relative to path.
   *
   * @param filename A string containing a name of a file
   * @param path A string containing a path
   *
   * @return The relative path from filename to path
   */
  public static string get_relative_path (string filename, string path)
  {
    File parent;
    File descendant;
    string relative_path;

    parent = File.new_for_path (path);
    descendant = File.new_for_path (filename);
    relative_path = parent.get_relative_path (descendant);
    if (relative_path == null)
    {
      relative_path = filename;
      debug ("Fail to set relative path for '%s' to '%s'", filename, path);
    }
    return relative_path;
  }

  /**
   * Gets the absolute path for filename.
   *
   * @param filename A string containing a name of a file
   * @param basedir The base directory, the current dir by default
   *
   * @return The absolute path
   */
  public static string get_absolute_path (string? filename, string? basedir = null)
  {
    string absolute_path;

    absolute_path = filename;
    if (filename != null)
    {
      if (!Path.is_absolute (filename))
      {
        string base_dir;

        if (basedir != null)
        {
          base_dir = basedir;
        }
        else
        {
          base_dir = Environment.get_current_dir ();
        }
        absolute_path = Path.build_filename (base_dir, filename);
        if (!FileUtils.test (absolute_path, FileTest.EXISTS))
        {
          if (filename.has_prefix ("file://"))
          {
            absolute_path = filename.substring (7);
          }
          else
          {
            absolute_path = filename;
            debug ("Fail to set absolute path for: %s", filename);
          }
        }
      }
    }
    return absolute_path;
  }

  /**
   * Get the filename extension
   *
   * @param filename The name of a file
   *
   * @return The extension
   */
  public static string get_extension (string filename)
  {
    int offset;
    string ext = null;

    offset = filename.last_index_of_char ('.');
    if (offset != -1)
    {
      ext = filename.substring (offset + 1).down ();
    }
    return ext;
  }

  /**
   * Remove the extension
   *
   * @param filename The name of a file
   *
   * @return The name of the file without extension
   */
  public static string skip_extension (string filename)
  {
    string basename;

    if (filename != "." && filename != "..")
    {
      int offset;

      offset = filename.last_index_of_char ('.');
      if (offset != -1)
      {
        basename = filename.substring (0, offset);
      }
      else
      {
        basename = filename;
      }
    }
    else
    {
      basename = filename;
    }
    return basename;
  }

  /**
   * Compare filenames
   */
  public static int cmp_filename (string path_a, string path_b)
  {
    int ret = 0;
    string name_a, name_b;
    bool a_is_dir, b_is_dir;

    name_a = Path.get_basename (path_a);
    name_b = Path.get_basename (path_b);

    a_is_dir = FileUtils.test (path_a, FileTest.IS_DIR);
    b_is_dir = FileUtils.test (path_b, FileTest.IS_DIR);

    if (a_is_dir && !b_is_dir)
    {
      ret = -1;
    }
    else if (!a_is_dir && b_is_dir)
    {
      ret = 1;
    }
    else
    {
      string basename_a, basename_b;

      basename_a = skip_extension (name_a);
      basename_b = skip_extension (name_b);
      ret = strcmp (basename_a, basename_b);
      if (ret == 0)
      {
        ret = strcmp (name_a, name_b);
      }
    }
    return ret;
  }

  public static int strcmp (string? a, string? b)
  {
    if (a == null && b == null)
    {
      return 0;
    }
    else if (a == null)
    {
      return -1;
    }
    else if (b == null)
    {
      return 1;
    }
    else
    {
      return GLib.strcmp (a, b);
    }
  }

  /**
   * Get the last modification time of a file
   *
   * @param filename The name of a file
   *
   * @return The last modification tome of the file
   */
  public static uint64 get_mtime (string filename) throws Error
  {
    File file = File.new_for_path (filename);
    FileInfo info = file.query_info (FILE_ATTRIBUTE_TIME_MODIFIED, 0, null);

    return info.get_attribute_uint64 (FILE_ATTRIBUTE_TIME_MODIFIED);
  }

  /**
   * Setup an io channel
   */
  public static void set_up_io_channel (int fd, IOCondition cond,
                                        bool nblock,
                                        IOFunc func) throws IOChannelError
  {
    IOChannel ioc;

    ioc = new IOChannel.unix_new (fd);
    if (nblock)
    {
      try
      {
        ioc.set_flags (IOFlags.NONBLOCK);
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
    ioc.set_close_on_unref (true);
    ioc.add_watch (cond, func);
  }

  /**
   * Remove the last \n character of a string
   *
   * @param s A string
   */
  public static void chop (ref string s)
  {
    if (s.has_suffix ("\n"))
    {
      s = s.substring (s.length - 1);
    }
  }

  /**
   * Test if a string is a number
   *
   * @param s A string
   *
   * @return true if the string represente a number
   */
  public static bool str_is_num (string s)
  {
    bool ret = true;

    for (int i = 0; i < s.length; i++)
    {
      if (!s[i].isdigit ())
      {
        ret = false;
        break;
      }
    }
    return ret;
  }

  /**
   * Test if a character is a separator
   *
   * @param ch A character
   *
   * @return true if the character is a separator
   */
  public static bool char_is_separator (unichar ch)
  {
    if (ch.isprint () && (ch.isalnum () || ch == '_'))
    {
      return false;
    }
    return true;
  }

  /**
   * Create a button with an small image
   *
   * @param image A image
   *
   * @return A new Gtk.Button
   */
  public static Gtk.Button create_small_button (Gtk.Image image)
  {
    Gtk.Button button = new Gtk.Button ();
    button.set_relief (Gtk.ReliefStyle.NONE);

    /* don't allow focus on the close button */
    button.set_focus_on_click (false);

    /* make it as small as possible */
    Gtk.RcStyle rcstyle = new Gtk.RcStyle ();
    rcstyle.xthickness = rcstyle.ythickness = 0;
    button.modify_style (rcstyle);

    image.show ();

    button.add (image);

    /* Set minimal size */
    button.style_set.connect ((s, style) => {
      int h, w;

      Gtk.icon_size_lookup_for_settings (s.get_settings (), Gtk.IconSize.MENU,
                                         out w, out h);
      s.set_size_request (w + 2, h + 2);
    });

    return button;
  }

  /**
   * Create a button with an small image from stock
   *
   * @param stock_id A stock id
   *
   * @return A new Gtk.Button
   */
  public static Gtk.Button create_small_button_from_stock (string stock_id)
  {
    Gtk.Image image = new Gtk.Image.from_stock (stock_id, Gtk.IconSize.MENU);
    return Utils.create_small_button (image);
  }

  /**
   * Get Gdk.Pixbuf form stock_id
   *
   * @param stock_id A stock id
   * @param size The size of the Gdk.Pixbuf
   *
   * @return A new Gdk.Pixbuf
   */
  public static Gdk.Pixbuf get_pixbuf_for_stock (string stock_id,
                                                 Gtk.IconSize size)
  {
    Gdk.Pixbuf pixbuf;

    Gtk.Invisible w = new Gtk.Invisible ();
    pixbuf = w.render_icon (stock_id, size, "vala");
    return pixbuf;
  }

  /**
   * Get Gdk.Pixbuf form file
   *
   * @param filename A image filename
   * @param size The size of the Gdk.Pixbuf
   *
   * @return A new Gdk.Pixbuf
   */
  public static Gdk.Pixbuf get_pixbuf_for_file (string filename,
                                                Gtk.IconSize size) throws Error
  {
    File file;
    Icon icon;
    FileInfo file_info;
    Gdk.Pixbuf ret = null;

    file = File.new_for_path (filename);
    file_info = file.query_info (FILE_ATTRIBUTE_STANDARD_ICON,
                                 FileQueryInfoFlags.NONE,
                                 null);

    if (file_info != null)
    {
	    icon = file_info.get_icon ();
      if (icon != null)
      {
        int width;
        Gtk.IconTheme theme;
        Gtk.IconInfo icon_info;

        theme = Gtk.IconTheme.get_default ();
        Gtk.icon_size_lookup (size, out width, null);

        icon_info = theme.lookup_by_gicon (icon, width,
                                           Gtk.IconLookupFlags.USE_BUILTIN);

        if (icon_info != null)
        {
          ret = icon_info.load_icon ();
        }
      }
    }
    return ret;
  }

  /**
   * Get the mime type of a file
   *
   * @param filename The name of a file
   *
   * @return The mime type
   */
  public static string get_mime_type (string? filename) throws Error
  {
    string mime;

    if (filename == null)
    {
      mime = "text/plain";
    }
    else
    {
      File file;
      FileInfo file_info;

      file = File.new_for_path (filename);
      file_info = file.query_info ("standard::*", FileQueryInfoFlags.NONE, null);
      mime = ContentType.get_mime_type (file_info.get_content_type ());
    }
    return mime;
  }

  /**
   * Process the GTK events
   */
  public static void process_gtk_events ()
  {
    while (Gtk.events_pending ())
    {
      Gtk.main_iteration ();
    }
  }

  /**
   * Remove the last element of a string
   *
   * @param s A string
   * @param sep The separator
   *
   * @return true if the last element was removed
   */
  public static bool remove_last_element (ref string s, char sep)
  {
    int offset;
    bool ret = false;

    offset = s.last_index_of_char (sep);
    if (offset != -1)
    {
      s = s.substring (0, offset);
      ret = true;
    }
    return ret;
  }

  /**
   * Get the last element of a string
   *
   * @param s A string
   * @param sep The separator
   *
   * @return The last element
   */
  public static string get_last_element (string s, char sep)
  {
    int offset;
    string ret;

    offset = s.last_index_of_char (sep);
    if (offset != -1)
    {
      ret = s.substring (offset + 1);
    }
    else
    {
      ret = s;
    }
    return ret;
  }

  /**
   * Replace home dir with tilde (~)
   *
   * @param uri An URI
   *
   * @return The transformed URI
   */
  public static string replace_home_dir_with_tilde (string uri)
  {
    string home;
    string short_uri;

    home = Environment.get_home_dir ();
    if (uri.has_prefix (home))
    {
      short_uri = "~" + uri.substring (home.length, uri.length - home.length);
    }
    else
    {
      short_uri = uri;
    }
    return short_uri;
  }

  /**
   * Return the PixBuf that represents the specified type of symbol, or NULL.
   */
  public static Gdk.Pixbuf get_symbol_pixbuf (string? type)
  {
    Gdk.Pixbuf pixbuf;

    if (type != null)
    {
      if (Utils.icons == null)
      {
        Utils.icons = new HashTable<string, Gdk.Pixbuf> (str_hash, str_equal);
      }
      pixbuf = Utils.icons.lookup (type);
      if (pixbuf == null)
      {
        try
        {
          string path;
          string filename;

          filename = "%s.png".printf (type);
          path = Path.build_filename (Config.PIXMAPS_DIR, "symbols", filename);
          pixbuf = new Gdk.Pixbuf.from_file (path);
          Utils.icons.insert (type, pixbuf);
        }
        catch (Error e)
        {
          debug ("Could not load pixbuf: %s\n", e.message);
          pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.MISSING_IMAGE,
                                               Gtk.IconSize.MENU);
        }
      }
    }
    else
    {
      pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.MISSING_IMAGE,
                                           Gtk.IconSize.MENU);
    }
    return pixbuf;
  }

  private static Gtk.IconFactory icon_factory = null;

  /**
   * Register a new stock id icon
   *
   * @param filename Icon file
   * @param stock_id Icon id
   */
  public static void register_icon (string filename, string stock_id) throws Error
  {
    Gdk.Pixbuf pixbuf;
    Gtk.IconSet icon_set;

    if (icon_factory == null)
    {
      icon_factory = new Gtk.IconFactory ();
      icon_factory.add_default ();
    }
    pixbuf = new Gdk.Pixbuf.from_file (filename);
    icon_set = new Gtk.IconSet.from_pixbuf (pixbuf);
    icon_factory.add (stock_id, icon_set);
  }
}

