/* source-buffer.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public class Valide.SourceBuffer : Gtk.SourceBuffer
{
  private string _path;
  /**
   * The path of the file
   */
  public string path
  {
    get
    {
      return this._path;
    }

    set
    {
      if (value != null)
      {
        Gtk.SourceLanguage lang;
        Gtk.SourceLanguageManager manager;

        manager = Gtk.SourceLanguageManager.get_default ();
        lang = manager.guess_language (value, null);
        this.set_language (lang);
      }
      this._path = value;
    }
  }

  /**
   * The encoding of the buffer
   */
  public string encoding
  {
    get;
    protected set;
  }

  /**
   * Return the content of the buffer
   *
   * @return The content of the buffer
   */
  public string get_buffer_contents ()
  {
    Gtk.TextIter start, end;

    this.get_bounds (out start, out end);
    return this.get_text (start, end, false);
  }

  /**
   * Return the entire file content in UTF-8 encoding
   *
   * @return The content of the file
   */
  public string get_contents () throws Error
  {
    string contents;

    contents = Encoding.to_utf8 (this.path, out this._encoding);
    return contents;
  }

  /**
   * Write the entire buffer content in file
   */
  public void set_contents (string? new_path = null) throws Error
  {
    string contents;

    contents = Encoding.from_utf8 (this, out this._encoding);
    if (new_path != null)
    {
      this.path = new_path;
    }
    FileUtils.set_contents (this.path, contents, -1);
  }

  /**
   * Get the last word before the corsor
   *
   * @param start_word The iter at begining of the word
   * @param start_word The iter at end of the word
   *
   * @return the last word
   */
  public string get_last_word_and_iter (out Gtk.TextIter start_word,
                                        out Gtk.TextIter end_word)
  {
    bool found;
    unichar ch;
    string text;
    bool no_doc_start;
    Gtk.TextIter actual;
    Gtk.TextIter start_iter;
    Gtk.TextMark insert_mark;

    insert_mark = this.get_insert ();
    this.get_iter_at_mark (out actual, insert_mark);

    start_iter = actual;
    found = false;

    do
    {
      ch = actual.get_char ();
      if (Utils.char_is_separator (ch))
      {
        break;
      }
    } while (actual.forward_char ());
    end_word = actual;

    while ((no_doc_start = start_iter.backward_char ()) == true)
    {
      ch = start_iter.get_char ();
      //TODO Do better
      if (Utils.char_is_separator (ch))
      {
        found = true;
        break;
      }
    }

    if (!no_doc_start)
    {
      this.get_start_iter (out start_iter);
      text = start_iter.get_text (actual);
    }
    else
    {
      if (found)
      {
        start_iter.forward_char ();
        text = start_iter.get_text (actual);
      }
      else
      {
        start_iter = actual;
        text = "";
      }
    }
    start_word = start_iter;
    return text;
  }

  /**
   * Get the last word before the cursor
   *
   * @return the last word
   */
  public string get_last_word ()
  {
    Gtk.TextIter end;
    Gtk.TextIter start;

    return this.get_last_word_and_iter (out start, out end);
  }

  /**
   * Replace the current work
   *
   * @param text The new text
   */
  public void replace_actual_word (string text)
  {
    Gtk.TextMark mark;
    Gtk.TextIter word_end;
    Gtk.TextIter word_start;

    this.begin_user_action ();
    this.get_last_word_and_iter (out word_start, out word_end);

    mark = this.create_mark ("temp_replace", word_start, true);
    this.delete (word_start, word_end);
    this.get_iter_at_mark (out word_start, mark);
    this.insert (word_start, text, -1);
    this.delete_mark (mark);
    this.end_user_action ();
  }

  /**
   * For compatibility with Gedit.Document
   */
  public string? get_uri ()
  {
    if (this.path != null)
    {
      return "file://%s".printf (this.path);
    }
    else
    {
      return null;
    }
  }

  /**
   * For compatibility with Gedit.Document
   *
   * @todo
   */
  public string get_short_name_for_display ()
  {
    return this.path;
  }

  /**
   * For compatibility with Gedit.Document
   */
  public bool is_untouched ()
  {
    return !this.get_modified ();
  }
}

