/* searching.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

private class Valide.OptionsWindow : Gtk.Window
{
  public Gtk.CheckButton check_casse;
  public Gtk.CheckButton check_word;

  private bool delete_event_cb (Gtk.Widget widget, Gdk.Event  event)
  {
    // Prevent the alt+F4 keys
    return true;
  }

  private bool focus_out_event_cb (Gtk.Widget sender, Gdk.EventFocus event)
  {
    this.hide ();
    return false;
  }

  construct
  {
    Gtk.VBox vbox;

    this.set_border_width (5);
    this.set_decorated (false);
    this.set_gravity (Gdk.Gravity.SOUTH_WEST);
    this.set_position (Gtk.WindowPosition.MOUSE);
    this.set_skip_pager_hint (true);
    this.set_skip_taskbar_hint (true);

    vbox = new Gtk.VBox (false, 0);
    this.add (vbox);

    this.check_casse = new Gtk.CheckButton.with_label (_("Match case"));
    this.check_casse.active = ConfigManager.get_instance ().get_boolean ("Search",
                                                                         "case-sensitive");
    vbox.pack_start (this.check_casse, false, false, 0);
    this.check_casse.toggled.connect ((s) => {
      ConfigManager.get_instance ().set_boolean ("Search", "case-sensitive",
                                                 s.active);
    });

    this.check_word = new Gtk.CheckButton.with_label (_("Match entire word only"));
    this.check_word.active = ConfigManager.get_instance ().get_boolean ("Search",
                                                                        "entire-word");
    vbox.pack_start (this.check_word, false, false, 0);
    this.check_word.toggled.connect ((s) => {
      ConfigManager.get_instance ().set_boolean ("Search", "entire-word", s.active);
    });

    // Connect signals
    this.delete_event.connect (this.delete_event_cb);
    this.focus_out_event.connect (this.focus_out_event_cb);
  }
}

/**
 * A widget for search/replace in a Valide.Document
 */
public class Valide.Searching : Gtk.VBox
{
  private bool reverse = false;
  private Gtk.HBox hbox_search;
  private Gtk.HBox hbox_replace;
  private Gtk.Entry entry_search;
  private Gtk.Entry entry_replace;
  private Gtk.Label message_label;
  private OptionsWindow options_window;
  private Gtk.ToggleButton toggle_button;

  /**
   * The attached document
   */
  public unowned Document document
  {
    get;
    construct;
  }

  /**
   * The text search
   */
  public string search_text
  {
    get { return this.entry_search.get_text (); }
    set { this.entry_search.set_text (value); }
  }

  /**
   * The replacement text
   */
  public string replace_text
  {
    get { return this.entry_replace.get_text (); }
    set { this.entry_replace.set_text (value); }
  }

  /**
   * The searching is case sensitive
   */
  public bool casse
  {
    get { return this.options_window.check_casse.get_active (); }
    set { this.options_window.check_casse.active = value; }
  }

  /**
   * Search only the whole word
   */
  public bool word
  {
    get { return this.options_window.check_word.get_active (); }
    set { this.options_window.check_word.active = value; }
  }

  private bool search ()
  {
    int count = 0;
    Gdk.Color color;
    Gtk.TextIter end;
    Gtk.TextIter iter;
    bool found = false;
    Gtk.TextIter start;
    Gtk.SourceBuffer buffer;
    Gtk.SourceSearchFlags flags;

    flags = Gtk.SourceSearchFlags.TEXT_ONLY;
    buffer = this.document.buffer;
    buffer.get_selection_bounds (null, out iter);

    if (!this.casse)
    {
      flags |= Gtk.SourceSearchFlags.CASE_INSENSITIVE;
    }

    start = Gtk.TextIter ();
    end = Gtk.TextIter ();

    while (count < 2)
    {
      if (!this.reverse)
      {
        found = Gtk.source_iter_forward_search (iter, this.search_text, flags,
                                                out start, out end, null);
      }
      else
      {
        if (count == 0)
        {
          iter.backward_word_start ();
        }
        found = Gtk.source_iter_backward_search (iter, this.search_text, flags,
                                                 out start, out end, null);
      }
      if (found && this.word)
      {
        found = start.starts_word () &&  end.ends_word ();
      }
      if (!found)
      {
        if (!this.reverse)
        {
          buffer.get_start_iter (out iter);
        }
        else
        {
          buffer.get_end_iter (out iter);
        }
        count++;
      }
      else
      {
        break;
      }
    }

    if (found)
    {
      buffer.place_cursor (start);
      buffer.move_mark_by_name ("selection_bound", end);
      this.document.text_view.scroll_to_mark (buffer.get_insert (), 0.25,
                                              false, 0.0, 0.0);
      Gdk.Color.parse ("white", out color);
    }
    else
    {
      buffer.place_cursor (iter);
       Gdk.Color.parse ("#FFBFBF", out color);
       this.show ();
    }
    this.entry_search.modify_base (Gtk.StateType.NORMAL, color);
    return found;
  }

  private bool key_press_event_cb (Gtk.Widget sender, Gdk.EventKey event)
  {
    bool catched = false;

    if (event.keyval == Gdk.KeySyms.Escape)
    {
      this.hide ();
      this.document.grab_focus ();
      catched = true;
    }
    else if (event.keyval == Gdk.KeySyms.Return)
    {
      if ((event.state & Gdk.ModifierType.SHIFT_MASK) == Gdk.ModifierType.SHIFT_MASK)
      {
        if (sender == this.entry_search)
        {
          this.find_prev ();
        }
        else
        {
          this.replace_prev ();
        }
      }
      else
      {
        if (sender == this.entry_search)
        {
          this.find_next ();
        }
        else
        {
          this.replace_next ();
        }
      }
    }
    return catched;
  }

  private void show_options ()
  {
    int x, y;
    int wx, wy;
    Gtk.Window parent;
    Gdk.Display display;

    parent = this.get_toplevel () as Gtk.Window;
    this.options_window.set_transient_for (parent);

    display = Gdk.Display.get_default ();
    display.get_pointer (null, out x, out y, null);
    this.options_window.show_all ();
    this.options_window.get_size (out wx, out wy);
    y -= wy;
    this.options_window.move (x, y);
  }

  private Gtk.Button create_button (string stock_id, string tooltip)
  {
    Gtk.Button button;
    Gtk.Image image;

    image = new Gtk.Image.from_stock (stock_id, Gtk.IconSize.BUTTON);
    button = new Gtk.Button ();
    button.add (image);
    button.set_relief (Gtk.ReliefStyle.NONE);
    button.set_tooltip_text (tooltip);
    button.can_focus = false;
    return button;
  }

  private void send_message (string message)
  {
    this.message_label.set_text (message);
    this.message_label.show ();
    Timeout.add (1000 * 5, () => {
      this.message_label.hide ();
      return false;
    });
  }

  /**
   * Create a new Valide.Searching
   *
   * @param document The attached document
   */
  public Searching (Document document)
  {
    Object (document: document);
  }

  construct
  {
    Gtk.HBox hbox;
    Gtk.Image image;
    Gtk.Button button;

    this.pack_start (new Gtk.HSeparator (), false, true, 0);

    /* Message box */
    this.message_label = new Gtk.Label ("");
    this.message_label.set_alignment (0.0f, 0.5f);
    /** @TODO Find a better placement
    this.pack_start (this.message_label, false, true, 0);*/

    hbox = new Gtk.HBox (false, 5);
    this.pack_start (hbox, false, false, 0);

    /* Search */
    this.hbox_search = new Gtk.HBox (false, 0);
    hbox.pack_start (this.hbox_search, false, false, 0);

    button = this.create_button (Gtk.Stock.CLOSE, _("Close"));
    button.clicked.connect (() => {
      this.hide ();
    });
    this.hbox_search.pack_start (button, false, false, 0);

    this.hbox_search.pack_start (new Gtk.Label (_("Find: ")), false, false, 0);
    this.entry_search = new Gtk.Entry ();
    this.entry_search.changed.connect (() => {
      this.find (false, false);
    });
    this.entry_search.key_press_event.connect (this.key_press_event_cb);
    this.hbox_search.pack_start (this.entry_search, false, true, 0);

    this.options_window = new OptionsWindow ();

    image = new Gtk.Image.from_stock (Gtk.Stock.PREFERENCES, Gtk.IconSize.BUTTON);
    this.toggle_button = new Gtk.ToggleButton ();
    this.toggle_button.add (image);
    this.toggle_button.set_relief (Gtk.ReliefStyle.NONE);
    this.toggle_button.set_tooltip_text (_("Options"));
    this.toggle_button.can_focus = false;
    this.options_window.hide.connect (() => {
      this.toggle_button.set_active (false);
    });
    this.toggle_button.toggled.connect ((s) => {
      if (s.active)
      {
        this.show_options ();
      }
      else
      {
        this.options_window.hide ();
      }
    });
    this.hbox_search.pack_start (this.toggle_button, false, false, 0);

    button = this.create_button (Gtk.Stock.GO_BACK, _("Find previous"));
    button.clicked.connect (() => {
      this.find_prev ();
    });
    this.hbox_search.pack_start (button, false, false, 0);

    button = this.create_button (Gtk.Stock.GO_FORWARD, _("Find next"));
    button.clicked.connect (() => {
      this.find_next ();
    });
    this.hbox_search.pack_start (button, false, false, 0);

    /* Replace */
    this.hbox_replace = new Gtk.HBox (false, 0);
    hbox.pack_start (this.hbox_replace, false, false, 0);

    this.hbox_replace.pack_start (new Gtk.Label (_("Replace with: ")), false, false, 0);
    this.entry_replace = new Gtk.Entry ();
    this.entry_replace.key_press_event.connect (this.key_press_event_cb);
    this.hbox_replace.pack_start (this.entry_replace, false, true, 0);

    button = this.create_button (Gtk.Stock.GO_BACK, _("Find and replace previous"));
    button.clicked.connect (() => {
      this.replace_prev ();
    });
    this.hbox_replace.pack_start (button, false, false, 0);

    button = this.create_button (Gtk.Stock.GO_FORWARD, _("Find and replace next"));
    button.clicked.connect (() => {
      this.replace_next ();
    });
    this.hbox_replace.pack_start (button, false, false, 0);

    button = new Gtk.Button.with_label (_("Replace all"));
    button.set_relief (Gtk.ReliefStyle.HALF);
    button.can_focus = false;
    button.clicked.connect (() => {
      this.replace_all ();
    });
    this.hbox_replace.pack_start (button, false, false, 0);
  }

  /**
   * Find the next word
   *
   * @param show_replace Show the replace dialog
   * @param show_dialog Show the dialog
   *
   * @return true if the word is found
   */
  public bool find (bool show_replace = false, bool show_dialog = true)
  {
    string t;
    bool found = false;

    t = this.document.selected_text;
    if (show_dialog || (t == "" && this.search_text == ""))
    {
      if (t != "")
      {
        this.search_text = t;
      }

      this.show_search ();
      if (show_replace)
      {
        this.show_replace ();
      }
    }
    else
    {
      found = this.search ();
    }
    return found;
  }

  /**
   * Find the next word
   *
   * @return true if the word is found
   */
  public bool find_next ()
  {
    this.reverse = false;
    return this.find (false, false);
  }

  /**
   * Find the previous word
   *
   * @return true if the word is found
   */
  public bool find_prev ()
  {
    this.reverse = true;
    return this.find (false, false);
  }

  /**
   * Replace the next word
   *
   * @param show_dialog Show the search dialog
   */
  public void replace (bool show_dialog = true)
  {
    this.find (true, show_dialog);
  }

  /**
   * Replace the next word
   *
   * @return true if the word is found
   */
  public bool replace_next ()
  {
    string text = "";
    Gtk.TextIter start;
    Gtk.TextIter end;
    Gtk.TextBuffer buffer;

    buffer = this.document.buffer;
    if (buffer.get_selection_bounds (out start, out end))
    {
      text = buffer.get_text (start, end, false);
    }

    if (text != "" && text == this.search_text)
    {
      buffer.begin_user_action ();
      buffer.delete (start, end);
      buffer.insert (start, this.replace_text, -1);
      buffer.end_user_action ();
    }
    return this.find_next ();
  }

  /**
   * Replace the previous word
   *
   * @return true if the word is found
   */
  public bool replace_prev ()
  {
    string text = "";
    Gtk.TextIter start;
    Gtk.TextIter end;
    Gtk.TextBuffer buffer;

    buffer = this.document.buffer;
    if (buffer.get_selection_bounds (out start, out end))
    {
      text = buffer.get_text (start, end, false);
    }

    if (text != "" && text == this.search_text)
    {
      buffer.begin_user_action ();
      buffer.delete (start, end);
      buffer.insert (start, this.replace_text, -1);
      buffer.end_user_action ();
    }
    return this.find_prev ();
  }

  /**
   * Replace the all words
   */
  public void replace_all ()
  {
    if (this.search_text != "" && this.search_text != this.replace_text)
    {
      bool found;
      uint count = 0;
      Gtk.TextIter end;
      Gtk.TextMark mark;
      Gtk.TextIter iter;
      Gtk.TextIter start;
      Gtk.TextBuffer buffer;
      Gtk.SourceSearchFlags flags;

      flags = Gtk.SourceSearchFlags.TEXT_ONLY;
      buffer = this.document.buffer;

      if (!this.casse)
      {
        flags |= Gtk.SourceSearchFlags.CASE_INSENSITIVE;
      }

      buffer.get_start_iter (out iter);
      buffer.begin_user_action ();
      while ((found = Gtk.source_iter_forward_search (iter, this.search_text,
                                                      flags, out start, out end,
                                                      null)))
      {
        if (found && this.word)
        {
          found = start.starts_word () &&  end.ends_word ();
        }
        if (found)
        {
          mark = buffer.create_mark ("search", end, false);
          buffer.delete (start, end);
          buffer.insert (start, this.replace_text, -1);
          buffer.get_iter_at_mark (out end, mark);
          buffer.delete_mark (mark);
          count++;
        }
        iter = end;
      }
      buffer.end_user_action ();
      if (count < 2)
      {
        this.send_message (_("Found and replaced %d occurrence").printf (count));
      }
      else
      {
        this.send_message (_("Found and replaced %d occurrences").printf (count));
      }
    }
  }

  /**
   * Show the search dialog
   */
  public void show_search ()
  {
    this.show_all ();
    this.message_label.hide ();
    this.hbox_replace.hide ();
    if (this.search_text != "")
    {
      this.entry_search.select_region (0, (int)this.search_text.length);
    }
    this.entry_search.grab_focus ();
  }

  /**
   * Show the replace dialog
   */
  public void show_replace ()
  {
    this.show_all ();
    this.message_label.hide ();
    if (this.search_text != "")
    {
      if (this.replace_text != "")
      {
        this.entry_replace.select_region (0, (int)this.replace_text.length);
      }
    }
    else
    {
      this.entry_search.grab_focus ();
    }
  }

  /**
   * @see Gtk.Widget.grab_focus
   */
  public override void grab_focus ()
  {
    this.entry_search.grab_focus ();
  }
}

