/* project.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public class Valide.Source : Object, GLib.YAML.Buildable
{
  public string path { get; set; }

  public Source (string path)
  {
    Object (path: path);
  }
}

public class Valide.VapiDir : Object, GLib.YAML.Buildable
{
  public string path { get; construct set; }

  public VapiDir (string path)
  {
    Object (path: path);
  }
}

public class Valide.Package : Object, GLib.YAML.Buildable
{
  public string name { get; construct set; }

  public Package (string name)
  {
    Object (name: name);
  }
}

/**
 * A class for project
 */
public class Valide.Project : Object, GLib.YAML.Buildable
{
  /**
   * The file extension of projects
   */
  public const string[] FILE_EXT = {"vide"};

  public string filename;

  /**
   * The builder manager
   */
  public BuilderManager builders;

  /**
   * The file list of the project
   */
  public List<Source> files;

  /**
   * The list of the packages
   */
  public List<Package> packages;

  /**
   * The list of the vapi directory
   */
  public List<VapiDir> vapi_dir;

  private static Type[] child_types = {
     typeof (Source),
     typeof (Package),
     typeof (VapiDir)
  };
  private static const string[] child_tags = {
    "files",
    "packages",
    "vapi_dir"
  };

  /**
   * The signal closed is emited when the project is closed
   */
  public signal void closed ();

  /**
   * The signal closed is emited when a file is removed from the project
   */
  public signal void removed_file ();

  /**
   * The signal closed is emited when a file is added from the project
   */
  public signal void added_file ();

  /**
   * The signal options_changed is emited when the project options changed
   */
  public signal void options_changed ();

  private string _path;
  /**
   * The path of the project
   */
  public string path
  {
    get
    {
      _path = Path.get_dirname (this.filename);
      return _path;
    }
  }

  /**
   * The name of the project
   */
  public string name { get; set; }

  /**
   * The author of the project
   */
  public string author { get; set; }

  /**
   * The version of the project
   */
  public string version { get; set; }

  /**
   * The license of the project
   */
  public string license { get; set; }

  /**
   * The name of the builder
   */
  public string builder { get; set; }

  /**
   * The list of the compiler options
   */
  public BuilderOptions builder_options { get; set; }

  /**
   * The list of the executable options
   */
  public ExecutableOptions executable_options { get; set; }

  private DocumentManager _documents;
  /**
   * The document manager
   */
  private DocumentManager documents
  {
    get
    {
      return this._documents;
    }

    set
    {
      if (this.documents != null)
      {
        this.documents.tab_state_changed.disconnect (this.document_changed);
      }
      this._documents = value;
      this.documents.tab_state_changed.connect (this.document_changed);
    }
  }

  private void xml_to_yml (string filename) throws Error
  {
    string contents;

    FileUtils.get_contents (filename, out contents);
    if (contents.has_prefix ("<?xml"))
    {
      MarkupParseContext context;
      MarkupParser parser = {
        /* start_element */
        (context, element_name, attribute_names, attribute_values) => {
          this.set_data ("current_element", element_name);
        },
        /* end_element */
        (context, element_name) => {
          this.set_data ("current_element", null);
        },
        /* text */
        (context, text, text_len) => {
          string property;

          property = this.get_data<string> ("current_element");
          if (property != null)
          {
            NativeBuilderOptions options;

            options = this.builder_options as NativeBuilderOptions;
            switch (property)
            {
              case "file":
                this.files.append (new Source (text));
              break;
              case "vapi":
                this.vapi_dir.append (new VapiDir (text));
              break;
              case "pkg":
                this.packages.append (new Package (text));
              break;
              case "name":
                this.name = text;
              break;
              case "author":
                this.author = text;
              break;
              case "version":
                this.version = text;
              break;
              case "license":
                this.license = text;
              break;
              case "other":
                options.other = text;
              break;
              case "debug":
                options.debug = true;
              break;
              case "disable_assert":
                options.disable_assert = true;
              break;
              case "disable_checking":
                options.disable_checking = true;
              break;
              case "disable_non_null":
                options.disable_non_null = true;
              break;
              case "quiet":
                options.quiet = true;
              break;
              case "save_temps":
                options.save_temps = true;
              break;
              case "thread":
                options.thread = true;
              break;
            }
          }
        },
        /* passthrough */
        null,
        /* error */
        null
      };
      this.filename = filename;
      this.builder_options = new NativeBuilderOptions ();
      context = new MarkupParseContext (parser,
                                        MarkupParseFlags.TREAT_CDATA_AS_TEXT,
                                        (void*)this, null);
      context.parse (contents, contents.length);
      context.end_parse ();
      this.save ();
    }
  }

  public bool exist (string filename)
  {
    bool ret = false;

    foreach (Source file in this.files)
    {
      if (file.path == filename)
      {
        ret = true;
        break;
      }
    }
    return ret;
  }

  private void document_changed (DocumentManager sender, Document document)
  {
    string filename;
    string old_filename;

    old_filename = document.get_data<string> ("old_path");
    foreach (Source file in this.files)
    {
      if (!Path.is_absolute (file.path))
      {
        filename = Path.build_filename (this.path, file.path);
      }
      else
      {
        filename = file.path;
      }
      if (filename == old_filename)
      {
        file.path = Utils.get_relative_path (document.path, this.path);
        this.added_file ();
        break;
      }
    }
  }

  private string to_string () throws Error
  {
    StringBuilder sb;
    GLib.YAML.Writer writer;

    sb = new StringBuilder("");
    writer = new GLib.YAML.Writer ("Valide");
    writer.stream_object (this, sb);
    return sb.str;
  }

  static construct
  {
    GLib.YAML.Buildable.register_type (typeof (Project), child_tags, child_types);
    GLib.YAML.Buildable.set_property_hint (typeof (Project), "path",
                                           GLib.YAML.Buildable.PropertyHint.SKIP);
  }

  construct
  {
    this.name = "";
    this.author = "";
    this.license = "";
    this.version = "";
    this.files = new List<Source> ();
    this.vapi_dir = new List<VapiDir> ();
    this.packages = new List<Package> ();
    this.executable_options = new ExecutableOptions ();
    this.added_file.connect ((s) => {
      this.save ();
    });
    this.removed_file.connect ((s) => {
      this.save ();
    });
  }

  /**
   * @see Glib.YAML.Buildable.add_child
   */
  public void add_child (GLib.YAML.Builder builder, Object child,
                         string? type) throws Error
  {
    if (type == "files")
    {
      this.files.prepend ((Source)child);
    }
    else if (type == "packages")
    {
      this.packages.prepend ((Package)child);
    }
    else if (type == "vapi_dir")
    {
      this.vapi_dir.prepend ((VapiDir)child);
    }
  }

  /**
   * @see Glib.YAML.Buildable.get_children
   */
  public List<unowned Object>? get_children (string? type)
  {
    if (type == "files")
    {
      return this.files.copy ();
    }
    else if (type == "packages")
    {
      return this.packages.copy ();
    }
    else if (type == "vapi_dir")
    {
      return this.vapi_dir.copy ();
    }
    return null;
  }

  /**
   * Create a new project
   *
   * @param filename The project file name
   * @param documents The document manager
   * @param builders The builder manager
   */
  public static Project _new_from_filename (string filename,
                                            DocumentManager documents,
                                            BuilderManager builders) throws Error
  {
    Project project;
    FileStream file;

    project = new Project ();
    project.xml_to_yml (filename);
    project = null;
    file = FileStream.open (filename, "r");
    if (file != null)
    {
      GLib.YAML.Builder builder;

      builder = new GLib.YAML.Builder ("Valide");
      project = builder.build_from_file (file) as Project;
      project.documents = documents;
      project.builders = builders;
      project.filename = filename;
    }
    else
    {
      throw new IOError.NOT_FOUND ("The file '%s' doesn't exist", filename); 
    }
    return project;
  }

  /**
   * Close the project
   */
  public void close ()
  {
    foreach (Source file in this.files)
    {
      int pos;
      string path;

      path = this.get_real_filename (file.path);

      if (this.documents.is_open (path, out pos))
      {
        Document document;

        document = this.documents.get_nth (pos);
        this.documents.remove_document (document);
      }
    }
    this.closed ();
  }

  /**
   * Save the project file
   *
   * Warning: Normally, you don't have call this function, the project to save
   * itself when its properties (options, files, ...) changed.
   */
  public void save ()
  {
    try
    {
      FileUtils.set_contents (this.filename, this.to_string ());
    }
    catch (Error e)
    {
      warning (_("Couldn't save the project: %s"), e.message);
    }
  }

  /**
   * Open a file
   *
   * @param filename The name of the file
   *
   * @return The new document
   */
  public Document open_file (string filename) throws Error
  {
    string path;
    Document document;

    path = this.get_real_filename (filename);
    if (FileUtils.test (path, FileTest.EXISTS))
    {
      document = this.documents.create (path);
    }
    else
    {
      throw new DocumentError.BAD_URI (_("The file %s doesn't exist!"), filename);
    }
    return document;
  }

  /**
   * Add a list of file in project
   *
   * @param files The list of files
   */
  public void add_file (SList<string>? files = null)
  {
    SList<string> f;

    if (files == null)
    {
      Gtk.FileChooserDialog dialog;

      dialog = new Gtk.FileChooserDialog (_("Add file"), null,
                                          Gtk.FileChooserAction.OPEN,
                                          Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                          Gtk.Stock.ADD, Gtk.ResponseType.ACCEPT);

      dialog.set_current_folder (this.path);
      dialog.set_local_only (true);
      dialog.select_multiple = true;

      if (dialog.run () == Gtk.ResponseType.ACCEPT)
      {
        f = dialog.get_filenames ();
        files = f;
      }
      dialog.destroy ();
    }
    if (files != null)
    {
      foreach (string file in files)
      {
        string filename;

        try
        {
          if (!FileUtils.test (file, FileTest.EXISTS))
          {
            FileUtils.set_contents (file, "", -1);
          }

          filename = Utils.get_relative_path (file, this.path);
          if (!this.exist (filename))
          {
            this.files.append (new Source (filename));
          }
          else
          {
            message (_("File '%s' already added!"), filename);
          }
        }
        catch (Error e)
        {
          warning (_("Couldn't create the file: %s"), file);
        }
      }
      this.added_file ();
    }
  }

  /**
   * Remove a list of file in project
   *
   * @param files The list of files
   */
  public void remove_file (List<string>? files = null)
  {
    ProjectDialogRemove dialog = null;
    unowned List<string> files_to_remove = null;

    if (files == null)
    {

      dialog = new ProjectDialogRemove (this);
      if (dialog.run () == Gtk.ResponseType.APPLY)
      {
        files_to_remove = dialog.selected_files;
      }
      dialog.hide ();
    }
    else
    {
      files_to_remove = files;
    }
    if (files_to_remove != null)
    {
      string msg;
      bool remove;
      int response;
      string source_path;
      Gtk.MessageDialog msg_dialog;

      if (files_to_remove.length () < 2)
      {
        msg = _("This will remove the file from the project. Also delete the file?");
      }
      else
      {
        msg = _("This will remove the files from the project. Also delete the files?");
      }
      msg_dialog = new Gtk.MessageDialog (null,
                                          Gtk.DialogFlags.MODAL,
                                          Gtk.MessageType.QUESTION,
                                          Gtk.ButtonsType.YES_NO,
                                          msg);
      msg_dialog.add_button (Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL);
      msg_dialog.show_all ();
      response = msg_dialog.run ();
      msg_dialog.destroy ();
      if (response == Gtk.ResponseType.CANCEL)
      {
        return;
      }
      remove = (response == Gtk.ResponseType.YES);
      foreach (string filename in files_to_remove)
      {
        foreach (unowned Source source in this.files)
        {
          source_path = Utils.get_absolute_path (source.path,
                                                 Path.get_dirname (this.filename));
          if (source_path == filename)
          {
            this.files.remove (source);
            if (remove)
            {
              int pos;

              if (this.documents.is_open (filename, out pos))
              {
                this.documents.get_nth (pos).close ();
              }
              if (FileUtils.unlink (filename) < 0)
              {
                warning (_("Couldn't remove the file '%s' from disk"), filename);
              }
            }
          }
        }
      }
      if (dialog != null)
      {
        dialog.destroy ();
      }
      this.removed_file ();
    }
  }

  /**
   * The project is compiled?
   */
  public bool is_compiled ()
  {
    return FileUtils.test (this.get_executable_name (), FileTest.EXISTS);
  }

  /**
   * The executable is up to date?
   */
  public bool is_uptodate ()
  {
    string filename;
    uint64 exec_mtime;
    uint64 file_mtime;
    bool uptodate = true;

    try
    {
      exec_mtime = Utils.get_mtime (this.get_executable_name ());
      foreach (Source file in this.files)
      {
        filename = this.get_real_filename (file.path);
        file_mtime = Utils.get_mtime (filename);
        if (file_mtime > exec_mtime)
        {
          uptodate = false;
          break;
        }
      }
    }
    catch (Error e)
    {
      debug (e.message);
      uptodate = false;
    }
    return uptodate;
  }

  /**
   * Show the project option dialog
   */
  public void option_dialog (Gtk.Window? parent = null)
  {
    ProjectDialogOptions dialog;

    dialog = new ProjectDialogOptions (this, parent);
    dialog.run ();
    dialog.destroy ();
  }

  /**
   * Configure the project
   *
   * @return Exit status
   */
  public int configure () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.configure ();
  }

  /**
   * Compile the project
   *
   * @return Exit status
   */
  public int build () throws BuilderError
  {
    int pos;
    int status = -1;
    bool saved = true;

    foreach (Source file in this.files)
    {
      string path;

      path = this.get_real_filename (file.path);
      if (this.documents.is_open (path, out pos))
      {
        Document document;

        document = this.documents.get_nth (pos);
         // MANUEL BACHMANN : path is now the second argument
        document.save (null, this.path);
        if (!document.is_save)
        {
          saved = false;
          break;
        }
      }
    }

    if (saved)
    {
      Builder builder;

      builder = this.builders.create_builder (this);
      status = builder.build ();
    }
    return status;
  }

  /**
   * Install the project
   *
   * @return Exit status
   */
  public int install () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.install ();
  }

  /**
   * Uninstall the project
   *
   * @return Exit status
   */
  public int uninstall () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.uninstall ();
  }

  /**
   * Distribute the project
   *
   * @return Exit status
   */
  public int dist () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.dist ();
  }

  /**
   * Clean the project
   *
   * @return Exit status
   */
  public int clean () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.clean ();
  }

  /**
   * Clean the project for distribution
   *
   * @return Exit status
   */
  public int distclean () throws BuilderError
  {
    Builder builder;

    builder = this.builders.create_builder (this);
    return builder.distclean ();
  }

  /**
   * Execute the project
   */
  public void execute ()
  {
    if (this.is_compiled ())
    {
      bool run = true;

      if (!this.is_uptodate ())
      {
        Gtk.MessageDialog dialog;

        dialog = new Gtk.MessageDialog (null, Gtk.DialogFlags.MODAL,
                                        Gtk.MessageType.QUESTION,
                                        Gtk.ButtonsType.YES_NO,
                                        _("The executable is too old, would you like really executing it?"));
        dialog.add_button ("Rebuild", Gtk.ResponseType.APPLY);
        switch (dialog.run ())
        {
          case Gtk.ResponseType.NO:
            run = false;
          break;
          case Gtk.ResponseType.APPLY:
            try
            {
              if (this.build () != 0)
              {
                run = false;
              }
            }
            catch (Error e)
            {
              run = false;
              warning (e.message);
            }
          break;
        }
        dialog.destroy ();
      }
      if (run)
      {
        Executable executable;
        ExecutableOptions options;

        executable = new Executable ();
        executable.executable = "'%s'".printf (this.get_executable_name ());
        options = this.executable_options;
        if (options.working_dir == "")
        {
          options.working_dir = this.path;
        }
        this.builders.executables.run (executable, options);
      }
    }
    else
    {
        // instead of asking to compile... compile !
      //warning (_("You should compile the project before executing it!"));

      bool run = true;

      try
      {
        if (this.build () != 0)
        {
          run = false;
        }
      }
      catch (Error e)
      {
        run = false;
        warning (e.message);
      }

      if (run)
      {
        Executable executable;
        ExecutableOptions options;

        executable = new Executable ();
        executable.executable = "'%s'".printf (this.get_executable_name ());
        options = this.executable_options;
        if (options.working_dir == "")
        {
          options.working_dir = this.path;
        }
        this.builders.executables.run (executable, options);
      }
    }
  }

  /**
   * Test if a filename is a project file.
   *
   * @param filename a filename.
   *
   * @return true if a filename is a project file.
   */
  public static bool accept_file (string filename)
  {
    bool accept = false;
    string ext;

    ext = Utils.get_extension (filename);
    foreach (string e in Project.FILE_EXT)
    {
      if (e == ext)
      {
        accept = true;
        break;
      }
    }
    return accept;
  }

  /**
   * Get the real file name
   *
   * @param filename A file include in this project
   *
   * @return The real name of the file
   */
  public string get_real_filename (string filename)
  {
    string path;

    if (!Path.is_absolute (filename))
    {
      path = Path.build_filename (this.path, filename);
      if (!FileUtils.test (path, FileTest.EXISTS))
      {
        path = null;
        foreach (Source file in this.files)
        {
          if (file.path.has_suffix (filename))
          {
            path = file.path;
            break;
          }
        }
        if (path == null)
        {
          path = filename;
        }
      }
    }
    else
    {
      path = filename;
    }
    return path;
  }

  /**
   * Return the complete path of the executable
   *
   * @return The name of the executable
   */
  public string get_executable_name ()
  {
    string exe_name;

    if (this.executable_options.program == "")
    {
      exe_name = Path.build_filename (this.path, this.name);
      if (Config.OS == "win32")
      {
        exe_name += ".exe";
      }
    }
    else
    {
      if (Path.is_absolute (this.executable_options.program))
      {
        exe_name = this.executable_options.program;
      }
      else
      {
        exe_name = Path.build_filename (this.path, this.executable_options.program);
      }
    }
    return exe_name;
  }
}

