/* project-manager.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 *

public errordomain Valide.ProjectError
{
  NO_PROJECT
}

/**
 * The project manager
 */
public class Valide.ProjectManager : Gtk.VBox
{
  private enum Col
  {
    ICON,
    NAME,
    PATH,
    NB
  }

  private Gtk.Menu popup;
  private Gtk.ComboBox combo_box;
  private Gtk.TreeView tree_view;
  private Gtk.TreeStore tree_store;
  private Gtk.ListStore list_store;
  private SList<string> expanded_rows;
  private Gtk.ScrolledWindow scrolled_window;

  /**
   * This signal was emitted when an project is opened
   *
   * @param project The emiter
   */
  public signal void project_opened (Project project);

  /**
   * This signal was emitted when an project is closed
   */
  public signal void project_closed (Project project);

  /**
   * is emitted when a project options changed
   *
   * @param project The emiter
   */
  public signal void project_options_changed (Project project);

  /**
   * The ui manager
   */
  public UIManager ui_manager
  {
    get;
    construct;
  }

  /**
   * The document manager
   */
  public DocumentManager documents
  {
    get;
    construct;
  }

  /**
   * The builder manager
   */
  public BuilderManager builders
  {
    get;
    construct;
  }

  private unowned Project _current;
  /**
   * The current project
   */
  public Project current
  {
    get
    {
      return this._current;
    }
    /* @warning Doesn't update the combo box */
    private set
    {
      this._current = value;
      this.refresh ();
    }
  }

  private bool is_expanded_row (string name)
  {
    bool ret = false;

    foreach (string i in this.expanded_rows)
    {
      if (i == name)
      {
        ret = true;
        break;
      }
    }
    return ret;
  }

  private void add_expanded_row (string name)
  {
    if (!this.is_expanded_row (name))
    {
      this.expanded_rows.prepend (name);
    }
  }

  private void remove_expanded_row (string name)
  {
    unowned SList<string> l;

    if (this.is_expanded_row (name) == true)
    {
      for (l = this.expanded_rows; l != null; l = l.next)
      {
        if (l.data == name)
        {
          this.expanded_rows.remove_all (l.data);
        }
      }
    }
  }

  /**
   * Re-expand any row when the list is repopulated if that row was previously
   * expanded.
   */
  private void restore_expanded_rows ()
  {
    Gtk.TreeIter iter;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    if (model.get_iter_first (out iter) == true)
    {
      for (bool r = model.get_iter_first (out iter); 
           r == true;
           r = model.iter_next (ref iter))
      {
        string name;

        model.get (iter, Col.NAME, out name,  -1);
        if (this.is_expanded_row (name))
        {
          Gtk.TreePath path = model.get_path (iter);
          this.tree_view.expand_row (path, false);
        }
      }
    }
  }

  private void on_row_expanded (Gtk.TreeView sender, Gtk.TreeIter iter,
                                Gtk.TreePath path)
  {
    string name;        
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get (iter, Col.NAME, out name, -1);
    return_if_fail (name != null);

    this.add_expanded_row (name);
  }

  private void on_row_collapsed (Gtk.TreeView sender, Gtk.TreeIter iter,
                                 Gtk.TreePath path)
  {
    string name;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get (iter, Col.NAME, out name, -1);
    return_if_fail (name != null);

    remove_expanded_row (name);
  }

  private string copy_template (Template template,
                                string path, string name, string author,
                                string version, string license,
                                string language, string builder) throws Error
  {
    string contents;
    Project project;

    project = new Project ();
    project.name = name;
    project.author = author;
    project.version = version;
    project.license = license;
    project.builder = builder;
    project.filename = Path.build_filename (path, name + "." + Project.FILE_EXT[0]);
    project.builder_options = template.builder_options;
    project.packages = template.packages.copy ();
    project.vapi_dir = template.vapi_dir.copy ();
    project.files = template.files.copy ();
    project.executable_options.working_dir = path;
    if (language == "vala")
    {
      project.files.concat (template.files_vala.copy ());
    }
    else if (language == "genie")
    {
      project.files.concat (template.files_genie.copy ());
    }

    // Copy source files
    foreach (Source source in project.files)
    {
      File src;
      File dst;

      src = File.new_for_path (Path.build_filename (template.path, source.path));
      dst = File.new_for_path (Path.build_filename (path, Path.get_basename (source.path)));
      src.copy (dst, FileCopyFlags.OVERWRITE, null, null);
    }

    // Copy license
    if (license != "None")
    {
      FileUtils.get_contents (license, out contents, null);

      if (Config.OS == "win32")
      {
        FileUtils.set_contents (Path.build_filename (path, "COPYING.TXT"), contents);
        project.files.append (new Source ("COPYING.TXT"));
      }
      else
      {
        FileUtils.set_contents (Path.build_filename (path, "COPYING"), contents);
        project.files.append (new Source ("COPYING"));
      }
    }
    project.save ();
    return project.filename;
  }

  private void file_select (Gtk.TreeView sender, Gtk.TreePath path,
                            Gtk.TreeViewColumn column)
  {
    Gtk.TreeIter iter;
    string filename = null;

    this.tree_store.get_iter (out iter, path);
    this.tree_store.get (iter, Col.PATH, out filename);
    if (!this.tree_store.iter_has_child (iter))
    {
      try
      {
        this.current.open_file (filename);
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      if (this.tree_view.is_row_expanded (path))
      {
        this.tree_view.collapse_row (path);
      }
      else
      {
        this.tree_view.expand_row (path, false);
      }
    }
  }

  private bool button_press (Gtk.Widget sender, Gdk.EventButton event)
  {
    if (event.type == Gdk.EventType.BUTTON_PRESS && event.button == 3)
    {
      this.popup.popup (null, null, null, event.button, event.time);
      return true;
    }
    return false;
  }

  private void setup_ui ()
  {
    bool active = false;

    if (this.current != null)
    {
      active = true;
    }
    this.ui_manager.get_action ("project-add-file").sensitive = active;
    this.ui_manager.get_action ("project-remove-file").sensitive = active;
    this.ui_manager.get_action ("project-options").sensitive = active;
    this.ui_manager.get_action ("project-configure").sensitive = active;
    this.ui_manager.get_action ("project-build").sensitive = active;
    this.ui_manager.get_action ("project-install").sensitive = active;
    this.ui_manager.get_action ("project-dist").sensitive = active;
    this.ui_manager.get_action ("project-clean").sensitive = active;
    this.ui_manager.get_action ("project-distclean").sensitive = active;
    this.ui_manager.get_action ("project-uninstall").sensitive = active;
    this.ui_manager.get_action ("project-execute").sensitive = active;
    this.ui_manager.get_action ("project-close").sensitive = active;
    this.sensitive = active;
  }

  private Gtk.TreeIter? search_iter (HashTable<string, Gtk.TreeIter?> iters,
                                     string path, Project project)
  {
    Gtk.TreeIter? iter = null;

    iter = iters.lookup (path);
    if (path != "/" & path != "." && iter == null)
    {
      Gdk.Pixbuf pixbuf;
      Gtk.TreeIter? root;

      root = this.search_iter (iters, Path.get_dirname (path), project);
      this.tree_store.append (out iter, root);
      pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.DIRECTORY,
                                           Gtk.IconSize.BUTTON);
      this.tree_store.set (iter, Col.ICON, pixbuf,
                           Col.NAME, Path.get_basename (path),
                           Col.PATH, Path.build_filename (project.path, path));
      iters.insert (path, iter);
    }
    return iter;
  }

  private void refresh ()
  {
    Project project;

    this.tree_store.clear ();
    project = this.current;
    if (project != null)
    {
      double current_pos;
      Gtk.VScrollbar vcrollbar;
      HashTable<string, Gtk.TreeIter?> dir_iters;

      vcrollbar = this.scrolled_window.get_vscrollbar () as Gtk.VScrollbar;
      current_pos = vcrollbar.get_value();

      dir_iters = new HashTable<string, Gtk.TreeIter?> (str_hash, str_equal);
      foreach (Source file in project.files)
      {
        string path;
        string filename;
        Gdk.Pixbuf pixbuf;
        Gtk.TreeIter iter;
        Gtk.TreeIter? root;

        path = project.get_real_filename (file.path);
        filename = Path.get_basename (path);

        root = this.search_iter (dir_iters, Path.get_dirname (file.path), project);
        this.tree_store.append (out iter, root);
        try
        {
          pixbuf = Utils.get_pixbuf_for_file (path, Gtk.IconSize.BUTTON);
        }
        catch (Error e)
        {
          pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.MISSING_IMAGE,
                                               Gtk.IconSize.BUTTON);
        }
        this.tree_store.set (iter, Col.NAME, filename, Col.ICON, pixbuf,
                             Col.PATH, path);
        this.restore_expanded_rows ();

        Utils.process_gtk_events ();
        vcrollbar.set_value(current_pos);
      }
    }
    this.setup_ui ();
  }

  private bool drag_motion_cb (Gtk.Widget sender,
                               Gdk.DragContext drag_context,
                               int x, int y,
                               uint time)
  {
    Gtk.TreePath path;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();
    /* Retrieve the drag location */
    this.tree_view.get_path_at_pos (x, y, out path, null, null, null);
    if (path != null)
    {
      selection.select_path (path);
    }
    else
    {
      selection.unselect_all ();
    }
    return true;
  }

  private void drag_data_received_cb (Gtk.Widget sender,
                                      Gdk.DragContext drag_context,
                                      int x, int y,
                                      Gtk.SelectionData data,
                                      uint info, uint time)
  {
    bool del = false;
    Project project;
    SList<string> files;
    Gtk.TreePath path;
    Gtk.TreeIter iter;
    string filename;
    string name;
    string project_name;
    string destination;

    /* Retrieve the drag location */
    this.tree_view.get_path_at_pos (x, y, out path, null, null, null);
    if (path == null)
    {
      project = this.current;
      destination = project.path;
    }
    else
    {
      this.tree_store.get_iter (out iter, path);
      this.tree_store.get (iter, Col.NAME, out project_name);
      this.tree_store.get (iter, Col.PATH, out filename);
      if (filename == null)
      {
        project = this.current;
        destination = project.path;
      }
      else
      {
        Gtk.TreeIter parent;

        if (!this.tree_store.iter_has_child (iter))
        {
          this.tree_store.iter_parent (out parent, iter);
          iter = parent;
        }
        this.tree_store.get (iter, Col.NAME, out name);

        this.tree_store.iter_parent (out parent, iter);
        this.tree_store.get (iter, Col.NAME, out project_name);
        project = this.current;

        destination = Path.build_filename (project.path, name);
      }
    }

    files = new SList<string> ();
    foreach (string uri in ((string)data.data).split ("\r\n"))
    {
      if (uri != "")
      {
        File src;
        string filepath;
        string basename;

        src = File.new_for_uri (uri);
        filepath = src.get_path ();
        basename = Path.get_basename (filepath);

        if (drag_context.suggested_action == Gdk.DragAction.LINK)
        {
          files.append (filepath);
        }
        else
        {
          try
          {
            File dest;
            string destpath;

            destpath = Path.build_filename (destination, src.get_basename ());
            dest = File.new_for_path (destpath);
            /* @TODO Ask for overwriting */
            src.copy (dest, FileCopyFlags.OVERWRITE, null, null);
            files.append (dest.get_path ());
            if (drag_context.suggested_action == Gdk.DragAction.MOVE)
            {
              src.delete (null);
              del = true;
            }
          }
          catch (Error e)
          {
            warning (e.message);
          }
          drag_context.suggested_action = drag_context.action;
        }
      }
    }
    if (files != null)
    {
      project.add_file (files);
    }
    Gtk.drag_finish (drag_context, true, del, time);
  }

  private int sort_func (Gtk.TreeModel model, Gtk.TreeIter a, Gtk.TreeIter b)
  {
    string path_a, path_b;

    model.get (a, Col.PATH, out path_a);
    model.get (b, Col.PATH, out path_b);

    return Utils.cmp_filename (path_a, path_b);
  }

  private bool is_open (string? filename, out Gtk.TreeIter iter)
  {
    bool opened = false;

    if (this.accept_file (filename))
    {
      Project project;

      this.list_store.get_iter_first (out iter);
      while (this.list_store.iter_is_valid (iter))
      {
        this.list_store.get (iter, 0, out project);
        if (project.filename == filename)
        {
          opened = true;
          break;
        }
        this.list_store.iter_next (ref iter);
      }
    }
    return opened;
  }

  private void project_changed ()
  {
    Project project;
    Gtk.TreeIter iter;

    this.combo_box.get_active_iter (out iter);
    if (this.list_store.iter_is_valid (iter))
    {
      this.list_store.get (iter, 0, out project);
      this.current = project;
    }
    else
    {
      this.current = null;
    }
  }

  private void add_new_file (DocumentManager sender, Document document)
  {
    if (document.is_new && this.current != null)
    {
      Gtk.Dialog dialog;
      bool adding_file = false;

      dialog = new Gtk.Dialog ();
      dialog.title = _("Create file");
      dialog.vbox.pack_start (new Gtk.Label (_("Would you like to add the file to this project?")), true, true, 0);
      dialog.add_button (Gtk.Stock.YES, Gtk.ResponseType.YES);
      dialog.add_button (Gtk.Stock.NO, Gtk.ResponseType.NO);
      dialog.show_all ();
      if (dialog.run () == Gtk.ResponseType.YES)
      {
        adding_file = true;
      }
      dialog.destroy ();

      if (adding_file)
      {
        document.buffer.set_modified (true);
        document.save (this.current.path);
        if (document.is_save)
        {
          SList<string> files;

          files = new SList<string> ();
          files.append (document.path);
          this.current.add_file (files);
        }
      }
    }
  }

  /**
   * Create a new project manager
   *
   * @param ui_manager The ui manager
   * @param documents The document manager
   */
  public ProjectManager (UIManager ui_manager, DocumentManager documents,
                         BuilderManager builders)
  {
    Object (ui_manager: ui_manager, documents: documents,
            builders: builders, width_request: 200, border_width: 2);
  }

  construct
  {
    Gtk.TreeViewColumn col;
    Gtk.CellRenderer renderer;
    Gtk.TargetEntry[] targets;

    this.list_store = new Gtk.ListStore (2, typeof (Project), typeof (string));

    this.combo_box = new Gtk.ComboBox.with_model (this.list_store);
    renderer = new Gtk.CellRendererText ();
    this.combo_box.pack_start (renderer, true);
    this.combo_box.set_attributes(renderer, "text", 1);
    this.combo_box.changed.connect (this.project_changed);
    this.pack_start (this.combo_box, false, true, 0);

    this.scrolled_window = new Gtk.ScrolledWindow (null, null);
    this.scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    this.scrolled_window.set_shadow_type (Gtk.ShadowType.IN);
    this.pack_start (this.scrolled_window, true, true, 0);

    this.tree_view = new Gtk.TreeView ();
    this.tree_view.headers_visible = false;
    this.scrolled_window.add (this.tree_view);
    this.tree_view.row_expanded.connect (this.on_row_expanded);
    this.tree_view.row_collapsed.connect (this.on_row_collapsed);

    this.tree_store = new Gtk.TreeStore (Col.NB, typeof (Gdk.Pixbuf),
                                         typeof (string), typeof(string));
    this.tree_store.set_sort_column_id (Col.NAME, Gtk.SortType.ASCENDING);
    this.tree_store.set_sort_func (Col.NAME, this.sort_func);
    this.tree_view.set_model (this.tree_store);

    col = new Gtk.TreeViewColumn ();
    col.set_title (_("Projects"));

    renderer = new Gtk.CellRendererPixbuf ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "pixbuf", Col.ICON);

    renderer = new Gtk.CellRendererText ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "markup", Col.NAME);

    this.tree_view.append_column (col);

    this.tree_view.row_activated.connect (this.file_select);
    /* Add popup */
    this.popup = this.ui_manager.get_widget ("/project-popup") as Gtk.Menu;
    this.tree_view.button_press_event.connect (this.button_press);

    this.refresh ();

    this.documents.tab_added.connect (this.add_new_file);
    /* Seting up the drag & drop */
    targets = new Gtk.TargetEntry[1];
    targets[0].target = "text/uri-list";
    targets[0].flags = 0;
    targets[0].info = 0;

    Gtk.drag_dest_set (this.tree_view, Gtk.DestDefaults.ALL, targets,
                       Gdk.DragAction.MOVE | Gdk.DragAction.COPY | Gdk.DragAction.LINK);
    this.tree_view.drag_motion.connect (drag_motion_cb);
    this.tree_view.drag_data_received.connect (drag_data_received_cb);
  }

  ~ProjectManager ()
  {
    this.documents.tab_added.disconnect (this.add_new_file);
  }

  /**
   * Create a new project
   *
   * @return true if the project is created
   */
  public bool create ()
  {
    bool created = false;
    ProjectDialog dialog;

    dialog = new ProjectDialog ();
    if (dialog.run () == Gtk.ResponseType.APPLY)
    {
      string path;

      path = dialog.project_dir;
      if (!FileUtils.test (path, FileTest.EXISTS))
      {
        Gtk.Dialog msg;

        msg = new Gtk.Dialog.with_buttons (_("New project"), null,
                                           Gtk.DialogFlags.MODAL,
                                           Gtk.Stock.NO, Gtk.ResponseType.NO,
                                           Gtk.Stock.YES, Gtk.ResponseType.YES,
                                           null);

        msg.vbox.pack_start (new Gtk.Label (_("Project directory doesn't exist. Would you like to create it?")),
                             true, true, 10);
        msg.show_all ();
        if (msg.run () == Gtk.ResponseType.YES)
        {
          DirUtils.create_with_parents (path, 0755);
          created = true;
        }
        msg.destroy ();
      }
      else
      {
        created = true;
      }

      if (created == true)
      {
        path = dialog.project_path;
        if (FileUtils.test (path, FileTest.EXISTS))
        {
          Gtk.Dialog msg;

          msg = new Gtk.Dialog.with_buttons (_("New project"), null,
                                             Gtk.DialogFlags.MODAL,
                                             Gtk.Stock.NO, Gtk.ResponseType.NO,
                                             Gtk.Stock.YES, Gtk.ResponseType.YES,
                                             null);

          msg.vbox.pack_start (new Gtk.Label (_("Directory already exists. Would you like to overwrite it?")),
                               true, true, 10);
          msg.show_all ();
          if (msg.run () == Gtk.ResponseType.NO)
          {
            created = false;
          }
          msg.destroy ();
        }
        else
        {
          DirUtils.create (path, 0755);
        }
      }

      if (created == true)
      {
        try
        {
          string prj;

          prj = this.copy_template (Template._new (dialog.template),
                                    path, dialog.project_name,
                                    dialog.project_author,
                                    dialog.project_version,
                                    dialog.project_license,
                                    dialog.lang,
                                    dialog.builder);
          this.open (prj);
        }
        catch (Error e)
        {
          warning (e.message);
          created = false;
        }
      }
    }
    dialog.destroy ();
    return created;
  }

  /**
   * Open a project
   *
   * @param filename The name of project file
   *
   * @return ture if the project is opened
   *
   * @throws Error
   */
  public bool open (string? filename = null) throws Error
  {
    string file_name;
    Gtk.TreeIter iter;
    bool opened = false;
    Gtk.FileFilter filter;
    Project project = null;

    file_name = Utils.get_absolute_path (filename);
    if (!this.is_open (file_name, out iter))
    {
      if (file_name == null)
      {
        Gtk.FileChooserDialog dialog;

        dialog = new Gtk.FileChooserDialog (_("Open project"), null,
                                            Gtk.FileChooserAction.OPEN,
                                            Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                            Gtk.Stock.OPEN, Gtk.ResponseType.ACCEPT,
                                            null);
        dialog.set_current_folder (ConfigManager.get_instance ().get_string ("Projects", "default-directory"));

          /* get last used save folder from config file,
          if blank use the user's home dir */
        string project_dir = ConfigManager.get_instance ().get_string ("Projects", "default-directory");
        if (project_dir != "")
          dialog.set_current_folder (project_dir);
        else
         dialog.set_current_folder (Environment.get_home_dir ());

        /* Add filter */
        filter = new Gtk.FileFilter ();
        filter.set_name (_("Val(a)IDE project file"));
        foreach (string ext in Project.FILE_EXT)
        {
          filter.add_pattern ("*." + ext);
        }
        dialog.add_filter (filter);

        if (dialog.run () == Gtk.ResponseType.ACCEPT)
        {
          file_name =  dialog.get_filename ();
        }
        dialog.destroy ();
      }
      if (file_name != null)
      {
        if (FileUtils.test (file_name, FileTest.EXISTS))
        {
          project = Project._new_from_filename (file_name, this.documents,
                                          this.builders);
          project.closed.connect (this.close);
          project.added_file.connect (this.refresh);
          project.removed_file.connect (this.refresh);
          project.options_changed.connect ((s) => {
            this.refresh ();
            this.project_options_changed (s);
          });

          this.list_store.append (out iter);
          this.list_store.set (iter, 0, project, 1, project.name);

          opened = true;
        }
        else
        {
          warning (_("The project file doesn't exist!"));
        }
      }
    }
    else
    {
      this.list_store.get (iter, 0, out project);
      opened = true;
    }
    if (opened)
    {
      this.combo_box.set_active_iter (iter);
      this.project_opened (project);

        // ferme tous les documents ouverts s'il y en a
      if (this.documents.get_n_pages () > 0)
      {
        this.documents.close_all ();
      }

        // ouvre "main.vala" s'il existe
      if (this.current.exist ("main.vala"))
      {
        this.current.open_file ("main.vala");
      }
    }
    return opened;
  }

  /**
   * Close the project
   *
   * @param project The project
   */
  public void close ()
  {
    if (this.current != null)
    {
      Project project;
      Gtk.TreeIter iter;

      project = this.current;
      if (this.is_open (project.filename, out iter))
      {
        this.list_store.remove (iter);
        this.project_closed (project);

        this.list_store.get_iter_first (out iter);
        if (this.list_store.iter_is_valid (iter))
        {
          this.combo_box.set_active_iter (iter);
        }
      }

       // NON ! Trouver un moyen de rcuprer les docs
      //this.documents.close_all ();
    }
    else
    {
      message (_("No project open!"));
    }
  }

  /**
   * Add a file in the current project
   */
  public void add_file ()
  {
    if (this.current != null)
    {
      this.current.add_file ();
    }
    else
    {
      message (_("No project open!"));
    }
  }

  /**
   * Remove file to the current project
   */
  public void remove_file ()
  {
    if (this.current != null)
    {
      this.current.remove_file ();
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void configure ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.configure ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void build ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.build ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void install ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.install ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void uninstall ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.uninstall ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void dist ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.dist ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void clean ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.clean ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void distclean ()
  {
    if (this.current != null)
    {
      try
      {
        this.current.distclean ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void execute ()
  {
    if (this.current != null)
    {
      this.current.execute ();
    }
    else
    {
      message (_("No project open!"));
    }
  }

  public void options ()
  {
    if (this.current != null)
    {
      this.current.option_dialog ();
    }
    else
    {
      message (_("No project open!"));
    }
  }

  /**
   * Open the selected file
   */
  public void open_selected_file ()
  {
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      Gtk.TreePath path;
      Gtk.TreeViewColumn column;

      path = this.tree_store.get_path (iter);
      column = this.tree_view.get_column (0);
      this.file_select (this.tree_view, path, column);
    }
  }

  /**
   * Remove the selected file
   */
  public void remove_selected_file ()
  {

    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      string filename;
      List<string> files;

      files = new List<string> ();
      this.tree_store.get (iter, Col.PATH, out filename);
      files.append (filename);
      this.current.remove_file (files);
    }
  }

  /**
   * Test if a filename is a project file.
   *
   * @param filename a filename.
   *
   * @return true if a filename is a project file.
   */
  public bool accept_file (string? filename)
  {
    bool accept = false;

    if (filename != null)
    {
      string ext;

      ext = Utils.get_extension (filename);
      accept = (ext in Project.FILE_EXT);
    }
    return accept;
  }
}
