/* project-dialog.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

private enum Valide.ColumnType
{
  ICON,
  NAME,
  DESCRIPTION,
  PATH,
  NB_COLUMNS
}

/**
 * A dialog box widget for create a new project
 */
public class Valide.ProjectDialog : AbstractProjectDialog
{
  private Gtk.ListStore list_store;
  private Gtk.ListStore licenses_list;

  /**
   * The apply signal was emetted when user apply the changements
   */
  public signal void apply ();

  /**
   * The name of the project
   */
  public string project_name
  {
    get
    {
      return this.widgets.name_entry.text;
    }
  }

  /**
   * The author of the project
   */
  public string project_author
  {
    get
    {
      return this.widgets.author_entry.text;
    }
  }

  /**
   * The version of the project
   */
  public string project_version
  {
    get
    {
      return this.widgets.version_entry.text;
    }
  }

  /**
   * The license of the project
   */
  private string _license;
  public string project_license
  {
    get
    {
      Gtk.TreeIter iter;

      _license = "None";
      if (this.widgets.licenses_combo.get_active_iter (out iter))
      {
        this.licenses_list.get (iter, 0, out _license);
      }
      return _license;
    }
  }

  /**
   * The directory of the project
   */
  private string _project_dir;
  public string project_dir
  {
    get
    {
      _project_dir = this.widgets.dir_chooser.get_filename ();
      return _project_dir;
    }
    set
    {
      this.widgets.dir_chooser.set_filename (value);
      _project_dir = value;
    }
  }

  /**
   * The path of the project
   */
  private string _project_path;
  public string project_path
  {
    get
    {
      _project_path = this.widgets.path_entry.get_text ();
      return _project_path;
    }
  }

  /**
   * The language of the project
   */
  private string _lang;
  public string lang
  {
    get
    {
      Gtk.TreeIter iter;
      Gtk.TreeModel model;

      _lang = "";
      if (this.widgets.lang_combo.get_active_iter (out iter))
      {
        model = this.widgets.lang_combo.get_model ();
        model.get (iter, 0, out _lang);
      }
      _lang = _lang.down();
      return _lang;
    }
  }

  /**
   * The builder of the project
   */
  private string _builder;
  public string builder
  {
    get
    {
      Gtk.TreeIter iter;
      Gtk.TreeModel model;

      _builder = "";
      if (this.widgets.builder.get_active_iter (out iter))
      {
        model = this.widgets.builder.get_model ();
        model.get (iter, 0, out _builder);
      }
      _builder = _builder.down();
      return _builder;
    }
  }

  /**
   * The project use for create the project
   */
  public string template
  {
    get;
    private set;
  }

  /**
   * The description of the template
   */
  public string project_desc
  {
    set
    {
      this.widgets.desc_label.set_text (value);
    }
  }

  private void populate_icon_view ()
  {
    List<string> files;

    files = TemplateManager.get_instance ().get_list ();
    this.list_store = new Gtk.ListStore (ColumnType.NB_COLUMNS,
                                         typeof (Gdk.Pixbuf),
                                         typeof (string),
                                         typeof (string),
                                         typeof (string));
    this.widgets.icon_view.set_model (this.list_store);
    this.widgets.icon_view.set_pixbuf_column (ColumnType.ICON);
    this.widgets.icon_view.set_text_column (ColumnType.NAME);

    foreach (string filename in files)
    {
      try
      {
        Gtk.TreeIter iter;
        Template template;
        Gdk.Pixbuf pixbuf;

        template = Template._new (filename);
        pixbuf = new Gdk.Pixbuf.from_file (Path.build_filename (template.path,
                                                                template.icon));
        this.list_store.append (out iter);
        this.list_store.set (iter, ColumnType.ICON, pixbuf,
                                   ColumnType.NAME, template.name,
                                   ColumnType.DESCRIPTION, template.description,
                                   ColumnType.PATH, template.path);
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
  }

  private void populate_license ()
  {
    List<string> files;
    Gtk.CellRendererText cell;
    string[] licenses_dir = { "/usr/share/common-licenses/",
                              "/usr/local/share/common-licenses/",
                              Path.build_filename (ConfigManager.get_instance ().get_home_dir (),
                                                   "licenses"),
                              "../share/common-licenses/"
                            };

    this.licenses_list = new Gtk.ListStore (2, typeof (string), typeof (string));
    this.widgets.licenses_combo.set_model (this.licenses_list);

    cell = new Gtk.CellRendererText ();
    this.widgets.licenses_combo.pack_start (cell, true);
    this.widgets.licenses_combo.set_attributes (cell, "text", 1);

    files = new List<string> ();
    foreach (string d in licenses_dir)
    {
      try
      {
        Dir dir;
        string filename;

        dir = Dir.open (d);
        while ((filename = dir.read_name ()) != null)
        {
          files.append (Path.build_filename (d, filename));
        }
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
    files.sort (strcmp);

    Gtk.TreeIter iter;
    this.licenses_list.append (out iter);
    this.licenses_list.set (iter, 0, "None", 1, "(None)");
     // we define the file name as "None", which is filtered
     // in "project-manager.vala"
     //
    foreach (string file in files)
    {
      this.licenses_list.append (out iter);
      this.licenses_list.set (iter, 0, file, 1, Path.get_basename (file));
    }
     //
    this.widgets.licenses_combo.set_active (0);
  }

  private void populate_builder ()
  {
    Gtk.TreeIter iter;
    Gtk.ListStore list_store;
    Gtk.CellRendererText cell;

    list_store = new Gtk.ListStore (1, typeof (string));
    list_store = this.widgets.builder.model as Gtk.ListStore ;
    cell = new Gtk.CellRendererText ();
    this.widgets.builder.pack_start (cell, true);
    this.widgets.builder.set_attributes (cell, "text", 0);
    foreach (string builder in BuilderManager.get_instance ().get_builders ())
    {
      list_store.append (out iter);
      list_store.set (iter, 0, builder);
    }
  }

  /**
   * This callback is called when the selected template changed
   */
  [CCode (instance_pos = -1)]
  protected void selection_changed (Gtk.IconView icon_view)
  {
    Gtk.TreeIter iter;
    List<Gtk.TreePath*> item;

    item = icon_view.get_selected_items ();
    if (item != null)
    {
      string tmp;
      Gtk.TreePath* path;
      Gtk.TreePath path2;

      path = item.data;
      path2 = new Gtk.TreePath.from_string (path->to_string ());
      this.list_store.get_iter (out iter, path2);

      this.list_store.get (iter, ColumnType.PATH, out tmp);
      this.template = Path.build_filename (tmp, "template.yml");

      this.list_store.get (iter, ColumnType.DESCRIPTION, out tmp);
      this.project_desc = tmp;
      this.widgets.assistant.set_page_complete (this.widgets.page2, true);
    }
    else
    {
      this.widgets.assistant.set_page_complete (this.widgets.page2, false);
    }
  }

  /**
   * The callback is called when the path of the project changed
   */
  [CCode (instance_pos = -1)]
  protected void project_path_change (Gtk.Widget sender)
  {
    this.widgets.path_entry.set_text (Path.build_filename (this.project_dir,
                                                           this.project_name) +  Path.DIR_SEPARATOR_S);

    if (this.project_name != "")
    {
      this.widgets.assistant.set_page_complete (this.widgets.page3, true);
    }
    else
    {
      this.widgets.assistant.set_page_complete (this.widgets.page3, false);
    }
  }

  construct
  {
    try
    {
      Gdk.Pixbuf icon;

      icon = new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "icone-32.xpm"));
      this.widgets.assistant.set_page_header_image (this.widgets.page1, icon);
      this.widgets.assistant.set_page_header_image (this.widgets.page2, icon);
      this.widgets.assistant.set_page_header_image (this.widgets.page3, icon);
    }
    catch (Error e)
    {
      debug (e.message);
    }

    this.populate_icon_view ();
    this.populate_license ();
    this.populate_builder ();
    this.connect_signals ("valide_project_dialog_");
    this.widgets.assistant.set_page_complete (this.widgets.page1, true);
    this.widgets.assistant.apply.connect (() => {
      this.apply ();
    });
    this.widgets.assistant.cancel.connect (() => {
      this.widgets.assistant.destroy ();
    });
    this.widgets.assistant.close.connect (() => {
      this.widgets.assistant.destroy ();
    });

    this.widgets.lang_combo.set_active (0);
    this.widgets.builder.set_active (0);

      // we cannot use "this.project_dir" because it's always "null" ???
    //this.project_dir = ConfigManager.get_instance ().get_string ("Projects", "default-directory");
    string project_dir = ConfigManager.get_instance ().get_string ("Projects", "default-directory");
      /* get project creation folder from config file,
      if blank use the user's home dir */
    if (project_dir != "")
      this.widgets.dir_chooser.set_current_folder (project_dir);
    else
      this.widgets.dir_chooser.set_current_folder (Environment.get_home_dir ());
  }

  /**
   * @see Gtk.Widget.show
   */
  public void show ()
  {
    this.widgets.assistant.show_all ();
  }

  /* Rewriting of the Gtk.Dialog.run function */
  private struct RunInfo
  {
    public MainLoop loop;
    public Gtk.ResponseType response_id;
    public bool destroyed;
  }

  private static void shutdown_loop (RunInfo ri)
  {
    if (ri.loop.is_running ())
    {
      ri.loop.quit ();
    }
  }

  private static void apply_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    ri.response_id = Gtk.ResponseType.APPLY;
    ProjectDialog.shutdown_loop (ri);
  }

  private static void cancel_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    ri.response_id = Gtk.ResponseType.CANCEL;
    ProjectDialog.shutdown_loop (ri);
  }

  private static void close_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    ri.response_id = Gtk.ResponseType.CLOSE;
    ProjectDialog.shutdown_loop (ri);
  }


  private static void unmap_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    ProjectDialog.shutdown_loop (ri);
  }

  private static bool delete_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    ProjectDialog.shutdown_loop (ri);
    return false;
  }

  private static void destroy_cb (Gtk.Widget sender, ref RunInfo ri)
  {
    /* shutdown_loop will be called by run_unmap_handler */
    ri.destroyed = true;
  }

  // doesn't work yet ; how to access "this"
  /*
  private static void restrictchars_cb (string text, int length, void* position)
  {
    if ( (text.contains ("/"))||(text.contains ("\\"))||(text.contains (":"))||(text.contains ("*"))||(text.contains ("?"))||(text.contains ("\""))||(text.contains ("<"))||(text.contains (">")) )
    {
      Signal.stop_emission_by_name (this.widgets.name_entry, "insert-text");
      Gdk.beep ();
    }
  }
  /*

  /**
   * @see Gtk.Dialog.run
   */
  public uint run ()
  {
    RunInfo ri = {null, Gtk.ResponseType.NONE, false};

    this.show ();
    Signal.connect (this.widgets.assistant, "apply", (Callback)apply_cb, (void*)(&ri));
    Signal.connect (this.widgets.assistant, "cancel", (Callback)cancel_cb, (void*)(&ri));
    Signal.connect (this.widgets.assistant, "close", (Callback)close_cb, (void*)(&ri));
    Signal.connect (this.widgets.assistant, "delete_event", (Callback)delete_cb, (void*)(&ri));
    Signal.connect (this.widgets.assistant, "unmap", (Callback)unmap_cb, (void*)(&ri));
    Signal.connect (this.widgets.assistant, "destroy", (Callback)destroy_cb, (void*)(&ri));
     /* prevent unallowed characters to be typed as project name */
    //Signal.connect (this.widgets.name_entry, "insert_text", (Callback)restrictchars_cb, (void*)(&ri));

    ri.loop = new MainLoop (null, false);
    Gdk.threads_leave ();
    ri.loop.run ();
    Gdk.threads_enter ();
    ri.loop = null;
    if (!ri.destroyed)
    {
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)apply_cb, (void*)(&ri));
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)cancel_cb, (void*)(&ri));
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)close_cb, (void*)(&ri));
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)delete_cb, (void*)(&ri));
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)unmap_cb, (void*)(&ri));
      SignalHandler.disconnect_by_func (this.widgets.assistant, (void*)destroy_cb, (void*)(&ri));
    }
    return ri.response_id;
  }

}

