/* executable.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public class Valide.EnvironmentVariable : Object, GLib.YAML.Buildable
{
  public string name { get; set; }
  public string value { get; set; }

  public EnvironmentVariable (string name, string value)
  {
    this.name = name;
    this.value = value;
  }
}

public class Valide.ExecutableOptions : Object, GLib.YAML.Buildable
{
  public List<EnvironmentVariable> environment;

  private static Type[] child_types= {
    typeof (EnvironmentVariable)
  };
  private static const string[] child_tags = {
    "environment"
  };

  public string program { get; set; }
  public string arguments { get; set; }
  public string working_dir { get; set; }

  static construct
  {
    GLib.YAML.Buildable.register_type (typeof (ExecutableOptions), child_tags, child_types);
  }

  construct
  {
    this.program = "";
    this.arguments = "";
    this.working_dir = "";
    this.environment = new List<EnvironmentVariable> ();
  }

  /**
   * @see Glib.YAML.Buildable.add_child
   */
  public void add_child (GLib.YAML.Builder builder, Object child,
                         string? type) throws Error
  {
    if (type == "environment")
    {
      this.environment.prepend ((EnvironmentVariable)child);
    }
  }

  /**
   * @see Glib.YAML.Buildable.get_children
   */
  public List<unowned Object>? get_children (string? type)
  {
    if (type == "environment")
    {
      return this.environment.copy ();
    }
    return null;
  }

  public string[] get_environment_variables ()
  {
    string[] environment = null;

    if (this.environment.length () > 0)
    {
      unowned string value;
      string[] default_env;
      HashTable<string, string> env;

      env = new HashTable<string, string> (str_hash, str_equal);

      /* Copy the default environment */
      default_env = Environment.list_variables ();
      foreach (string name in default_env)
      {
        value = Environment.get_variable (name);
        if (value != null)
        {
          env.insert (name, value);
        }
      }

      /* Add the user defined environment */
      try
      {
        Regex regex;

        regex = new Regex ("\\$[^: \\$]*");
        foreach (EnvironmentVariable variable in this.environment)
        {
          string v;

          if (variable.value.index_of ("$") != -1)
          {
            v = regex.replace_eval (variable.value, -1, 0, 0, (info, res) => {
              string match = info.fetch (0);
              if (match != "")
              {
                string r = env.lookup (match.substring (1)) ?? "";
                res.append (r);
              }
              return false;
            });
          }
          else
          {
            v = variable.value;
          }
          env.insert (variable.name, v);
        }
      }
      catch (Error e)
      {
        debug (e.message);
      }

      int i = 0;
      environment = new string[env.size () + 1];
      env.foreach ((key, value) => {
        environment[i] = "%s=%s\n".printf ((string)key, (string)value);
        i++;
      });
      environment[i] = null;
    }
    return environment;
  }
}

public class Valide.ExecutablePreferences : AbstractExecutablePreferences
{
  private enum Col
  {
    NAME,
    VALUE,
    NB
  }

  private ExecutableOptions options;
  private Gtk.ListStore list_store;

  public Project project
  {
    get;
    construct;
  }

  public Gtk.Widget widget
  {
    get
    {
      return this.widgets.table;
    }
  }

  private void update_environment ()
  {
    Gtk.TreeIter iter;

    this.options.environment = new List<EnvironmentVariable> ();
    if (this.list_store.get_iter_first (out iter))
    {
      string name;
      string value;

      do
      {
        this.list_store.get (iter, Col.NAME, out name, Col.VALUE, out value);
        this.options.environment.append (new EnvironmentVariable (name, value));
      } while (this.list_store.iter_next (ref iter));
    }
    this.project.save ();
  }

  private void on_name_edit (Gtk.CellRenderer sender, string path, string new_text)
  {
    Gtk.TreeIter iter;

    this.list_store.get_iter_from_string (out iter, path);
    this.list_store.set (iter, Col.NAME, new_text);
    this.update_environment ();
  }

  private void on_value_edit (Gtk.CellRenderer sender, string path, string new_text)
  {
    Gtk.TreeIter iter;

    this.list_store.get_iter_from_string (out iter, path);
    this.list_store.set (iter, Col.VALUE, new_text);
    this.update_environment ();
  }

  [CCode (instance_pos = -1)]
  protected void add_variable (Gtk.Button sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreePath path;
    Gtk.TreeViewColumn column;

    this.list_store.append (out iter);
    this.list_store.set (iter, Col.NAME, "", Col.VALUE, "");

    path = this.list_store.get_path (iter);
    column = this.widgets.tree_view.get_column (Col.NAME);
    this.widgets.tree_view.scroll_to_cell (path, column, false, 0, 0);
    this.widgets.tree_view.set_cursor (path, column ,true);
  }

  [CCode (instance_pos = -1)]
  protected void modify_variable (Gtk.Button sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.widgets.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      Gtk.TreePath path;
      Gtk.TreeViewColumn column;

      path = this.list_store.get_path (iter);
      column = this.widgets.tree_view.get_column (Col.VALUE);
      this.widgets.tree_view.scroll_to_cell (path, column, false, 0, 0);
      this.widgets.tree_view.set_cursor (path, column ,true);
    }
  }

  [CCode (instance_pos = -1)]
  protected void delete_variable (Gtk.Button sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.widgets.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      this.list_store.remove (iter);
      this.update_environment ();
    }
  }

  [CCode (instance_pos = -1)]
  protected void on_working_dir_changed (Gtk.Button sender)
  {
    Gtk.FileChooserDialog dialog;

    dialog = new Gtk.FileChooserDialog (_("Choose the working directory"), null,
                                       Gtk.FileChooserAction.OPEN,
                                       Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                       Gtk.Stock.OK, Gtk.ResponseType.OK);
    dialog.set_filename (this.widgets.executable_working_dir.get_text ());
    if (dialog.run () == Gtk.ResponseType.OK)
    {
      this.widgets.executable_working_dir.set_text (dialog.get_filename ());
    }
    dialog.destroy ();
  }

  [CCode (instance_pos = -1)]
  protected void on_program_changed (Gtk.Button sender)
  {
    Gtk.FileChooserDialog dialog;

    dialog = new Gtk.FileChooserDialog (_("Choose the executable"), null,
                                       Gtk.FileChooserAction.OPEN,
                                       Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                       Gtk.Stock.OK, Gtk.ResponseType.OK);
    dialog.set_filename (this.widgets.executable_program.get_text ());
    if (dialog.run () == Gtk.ResponseType.OK)
    {
      this.widgets.executable_program.set_text (dialog.get_filename ());
    }
    dialog.destroy ();
  }

  public ExecutablePreferences (Project project)
  {
    Object (project: project);
  }

  construct
  {
    Gtk.TreeIter iter;
    Gtk.TreeViewColumn column;
    Gtk.CellRendererText renderer;

    this.options = this.project.executable_options;

    if (this.options.program != null)
    {
      this.widgets.executable_program.set_text (this.options.program);
    }
    this.widgets.executable_program.changed.connect ((s) => {
      this.options.program = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    if (this.options.arguments != null)
    {
      this.widgets.executable_arguments.set_text (this.options.arguments);
    }
    this.widgets.executable_arguments.changed.connect ((s) => {
      this.options.arguments = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    if (this.options.working_dir != null)
    {
      this.widgets.executable_working_dir.set_text (this.options.working_dir);
    }
    this.widgets.executable_working_dir.changed.connect ((s) => {
      this.options.working_dir = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    this.list_store = new Gtk.ListStore (Col.NB, typeof (string),
                                                 typeof (string));
    this.widgets.tree_view.set_model (this.list_store);

    column = new Gtk.TreeViewColumn ();
    column.set_title (_("Name"));
    renderer = new Gtk.CellRendererText ();
    renderer.editable = true;
    renderer.edited.connect (this.on_name_edit);
    column.pack_start (renderer, true);
    column.set_attributes (renderer, "text", Col.NAME);
    this.widgets.tree_view.append_column (column);

    column = new Gtk.TreeViewColumn ();
    column.set_title (_("Value"));
    renderer = new Gtk.CellRendererText ();
    renderer.editable = true;
    renderer.edited.connect (this.on_value_edit);
    column.pack_start (renderer, true);
    column.set_attributes (renderer, "text", Col.VALUE);
    this.widgets.tree_view.append_column (column);

    foreach (EnvironmentVariable variable in this.options.environment)
    {
      this.list_store.append (out iter);
      this.list_store.set (iter, Col.NAME, variable.name,
                                 Col.VALUE, variable.value);
    }

    this.connect_signals ("valide_executable_preferences_");
  }
}

/**
 * A class representing an program can launch by #ExecutableManager.
 */
public class Valide.Executable : Object
{
  private static ExecutablePreferences preferences_widget;

  /**
   * An executable name.
   */
  public virtual string executable
  {
    get;
    set;
  }

  /**
   * A message show when executable was launched.
   *
   * You can use $cmd for show the command.
   */
  protected virtual string get_start_msg ()
  {
      return "";
  }

  /**
   * A message show when executable stop.
   *
   * You can use :
   *   * $status the exit status of the command.
   *   * $time the execution time
   */
  protected virtual string get_end_msg ()
  {
      return _("<b>Process return {$status}  execution time: {$time} s</b>");
  }

 /**
   * This function was called before adding line in ExecutableManager tree view.
   * You can modify it.
   *
   * @param line the text.
   * @param color the color of the text
   */
  public virtual void get_color_line (ref string line, out Gdk.Color color)
  {
    if (line.has_prefix ("**") && line.index_of ("DEBUG:") != -1)
    {
      Gdk.Color.parse (Utils.Color.INFO, out color);
    }
    else if (line.index_of ("WARNING **:") != -1)
    {
      Gdk.Color.parse (Utils.Color.WARNING, out color);
    }
    else if (line.index_of ("CRITICAL **:") != -1)
    {
      Gdk.Color.parse (Utils.Color.ERROR, out color);
    }
    else if (line.has_prefix ("** ERROR **:"))
    {
      Gdk.Color.parse (Utils.Color.ERROR, out color);
    }
    else
    {
      Gdk.Color.parse (Utils.Color.DEFAULT, out color);
    }
  }

  /**
   * This function was called when line in ExecutableManager was activated.
   *
   * @param line the line was actived.
   */
  public virtual void line_activated (string line)
  {
  }

  /**
   * This function is called when executable is launched
   *
   * @param cmd The command line
   *
   * @return A text
   */
  public string cmd_start (string cmd)
  {
    string msg;

    msg = this.get_start_msg ();
    msg = msg.replace ("{$cmd}", cmd);
    return msg;
  }

  /**
   * This function is called when executable quit
   *
   * @param status The exit status
   * @param time The enlapsed time
   *
   * @return A text
   */
  public string cmd_end (int status, double time)
  {
    string msg;

    msg = this.get_end_msg ();
    msg = msg.replace ("{$status}", status.to_string ());
    msg = msg.replace ("{$time}", "%.2f".printf (time));
    return msg;
  }

  public static Gtk.Widget preferences (Project project)
  {
    preferences_widget = new ExecutablePreferences (project);
    return preferences_widget.widget;
  }
}

