/* executable-manager.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public errordomain Valide.ExecutableError
{
  NO_EXECUTABLE
}

/**
 * Manager for launch external programs
 */
public class Valide.ExecutableManager : Gtk.HBox
{
  private enum Col
  {
    TEXT,
    COLOR,
    NB
  }

  private Pid pid;
  private int status;
  private Timer timer;
  private MainLoop loop;
  private Gtk.TreeView tree_view;
  private Gtk.ListStore list_store;

  /**
   * The signal run_exec is emited when an executable is launched
   */
  public signal void run_exec (Executable executable);

  /**
   * The UI manager
   */
  public UIManager ui_manager
  {
    get;
    construct;
  }

  private Executable current
  {
    get;
    set;
  }

  private bool running
  {
    get;
    set;
    default = false;
  }

  private string _text;
  private string text
  {
    get
    {
      _text = null;
      StringBuilder sb = new StringBuilder("");
      Gtk.TreeIter iter;

      if (this.list_store.get_iter_first (out iter) == true)
      {
        string val = null;

        do
        {
          this.list_store.get (iter, Col.TEXT, out val);
          sb.append(val);
          sb.append_unichar('\n');
        } while (this.list_store.iter_next (ref iter));
      }

      _text = sb.str;
      return _text;
    }
  }

  private void setup_ui ()
  {
    bool active = false;

    active = this.running;
    this.ui_manager.get_action ("executable-stop").sensitive = active;
  }

  private void clear ()
  {
    this.list_store.clear ();
  }

  private void copy ()
  {
    string str;
    unowned Gtk.Clipboard clipboard;

    clipboard = Gtk.Clipboard.get (Gdk.SELECTION_CLIPBOARD);
    str = this.text;

    if (str != "")
    {
      clipboard.set_text (str, -1);
    }
    else
    {
      warning (_("Nothing to copy!"));
    }
  }

  private void previous_line ()
  {
    Gtk.TreeIter iter;
    Gtk.TreePath path;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();
    if (!selection.get_selected (null, out iter)
        && !this.list_store.get_iter_first (out iter))
    {
      debug("the output line list is empty.");
      return;
    }

    path = this.list_store.get_path (iter);
    if (path.prev ())
    {
      this.list_store.get_iter (out iter, path);
    }
    else
    {
      int children;

      children = this.list_store.iter_n_children (null);
      if (children > 0)
      {
        this.list_store.iter_nth_child (out iter, null, children - 1);
      }
    }

    path = this.list_store.get_path (iter);
    if (this.list_store.iter_is_valid (iter))
    {
      Gtk.TreeViewColumn column;

      selection.select_iter (iter);
      column = tree_view.get_column (Col.TEXT);
      this.tree_view.scroll_to_cell (path, null, false, 0, 0);
      tree_view.row_activated (path, column);
    }
  }

  private void next_line ()
  {
    Gtk.TreeIter iter;
    Gtk.TreePath path;
    Gtk.TreeViewColumn column;
    Gtk.TreeSelection selection;

    selection = this.tree_view.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      if (!this.list_store.iter_next (ref iter))
      {
        this.list_store.get_iter_first (out iter);
      }
    }
    else
    {
      this.list_store.get_iter_first (out iter);
    }

    if (this.list_store.iter_is_valid (iter))
    {
      selection.select_iter (iter);
      path = this.list_store.get_path (iter);
      column = tree_view.get_column (Col.TEXT);
      this.tree_view.scroll_to_cell (path, null, false, 0, 0);
      tree_view.row_activated (path, column);
    }
  }

  private void row_activated_cb (Gtk.TreeView sender, Gtk.TreePath path,
                                 Gtk.TreeViewColumn column)
  {
    string line;
    Gtk.TreeIter iter;

    sender.model.get_iter (out iter, path);
    sender.model.get (iter, Col.TEXT, out line);
    this.current.line_activated (line);
  }

  private void add_line_markup (string line)
  {
    try
    {
      Regex regex;
      string line2;

      regex = new Regex ("[\n\r]*$",
                         RegexCompileFlags.OPTIMIZE,
                         RegexMatchFlags.NOTEMPTY);

      line2 = regex.replace (line, -1, 0, "", RegexMatchFlags.NOTEMPTY);

      if (line2.length > 0)
      {
        string[] lines;

        lines = line2.split ("\n");
        foreach (string l in lines)
        {
          Gdk.Color color;
          Gtk.TreeIter iter;
          Gtk.TreePath path;

          this.current.get_color_line (ref l, out color);

          this.list_store.append (out iter);
          this.list_store.set (iter, Col.TEXT, l, Col.COLOR, color);

          path = this.list_store.get_path (iter);
          this.tree_view.scroll_to_cell (path, null, false, 0, 0);
        }
      }
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  private void add_line (string line)
  {
    string line2;

    line2 = line.replace ("<", "&lt;");
    line2 = line2.replace (">", "&gt;");
    this.add_line_markup (line2);
  }

  private bool build_iofunc (IOChannel ioc, IOCondition cond)
  {
    bool ret = false;

    if ((cond & (IOCondition.IN | IOCondition.PRI)) != 0)
    {
      string msg;
      IOStatus rv;

      try
      {
        rv = ioc.read_line (out msg, null, null);
        if (msg != null)
        {
          this.add_line (msg);
        }
        if (rv == IOStatus.NORMAL || rv == IOStatus.AGAIN)
        {
          ret = true;
        }
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
    return ret;
  }

  private void cmd_watch (Pid pid, int status)
  {
    if (this.loop.is_running ())
    {
      this.running = false;
      this.timer.stop ();
      this.add_line_markup (this.current.cmd_end (status, this.timer.elapsed (null)));
      this.setup_ui ();
      this.status = status;
      this.loop.quit ();
    }
  }

  /**
   * Create a new Valide.ExecutableManager
   *
   * @param ui_manager The ui manager
   */
  public ExecutableManager (UIManager ui_manager)
  {
    Object (ui_manager: ui_manager);
  }

  construct
  {
    Gtk.VBox vbox;
    Gtk.Button button;
    Gtk.HSeparator separator;
    Gtk.TreeViewColumn column;
    Gtk.CellRendererText render;
    Gtk.ScrolledWindow scrolled_window;

    this.timer = new Timer ();

    vbox = new Gtk.VBox (false, 0);
    this.pack_start (vbox, false, false, 0);

    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.CLEAR,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Clear"));
    button.clicked.connect (this.clear);
    vbox.pack_start (button, false, false, 0);

    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.COPY,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Copy"));
    button.clicked.connect (this.copy);
    vbox.pack_start (button, false, false, 0);

    separator = new Gtk.HSeparator ();
    vbox.pack_start (separator, false, false, 0);

    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.GO_UP,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Previous line"));
    button.clicked.connect (this.previous_line);
    vbox.pack_start (button, false, false, 0);

    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.GO_DOWN,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Next line"));
    button.clicked.connect (this.next_line);
    vbox.pack_start (button, false, false, 0);

    scrolled_window = new Gtk.ScrolledWindow (null, null);
    scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    this.pack_start (scrolled_window, true, true, 0);

    this.list_store = new Gtk.ListStore (Col.NB, typeof (string), typeof (Gdk.Color));
    this.tree_view = new Gtk.TreeView.with_model (this.list_store);
    this.tree_view.row_activated.connect (this.row_activated_cb);
    scrolled_window.add (this.tree_view);

    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Message"), render,
                                                     "markup", Col.TEXT,
                                                     "foreground-gdk", Col.COLOR);
    this.tree_view.append_column (column);
    this.tree_view.headers_visible = false;

    this.notify["sensitive"].connect (this.clear);

    this.setup_ui ();
  }

  /**
   * Launch an executable
   *
   * @param executable An executable
   * @param options The executable optons
   *
   * @return Exit status
   */
  public int run (Executable executable, ExecutableOptions? options)
  {
    if (!this.running)
    {
      try
      {
        string cmd;
        int std_out;
        int std_err;
        string[] args;
        IOCondition cond;
        ExecutableOptions exec_options;

        if (options == null)
        {
          exec_options = new ExecutableOptions ();
        }
        else
        {
          exec_options = options;
        }
        this.current = executable;
        this.running = true;
        this.clear ();

        cmd = "%s %s".printf (executable.executable, exec_options.arguments);
        Shell.parse_argv (cmd, out args);
        this.setup_ui ();
        this.add_line_markup (executable.cmd_start (cmd));

        this.run_exec (executable);
        this.timer.start ();
        Process.spawn_async_with_pipes (exec_options.working_dir, args,
                                        exec_options.get_environment_variables (),
                                        SpawnFlags.SEARCH_PATH | SpawnFlags.DO_NOT_REAP_CHILD,
                                        null, out this.pid, null, out std_out,
                                        out std_err);
        this.setup_ui ();
        cond = IOCondition.IN | IOCondition.PRI | IOCondition.ERR
               | IOCondition.HUP | IOCondition.NVAL;
        Utils.set_up_io_channel (std_out, cond, true, build_iofunc);
        Utils.set_up_io_channel (std_err, cond, true, build_iofunc);
        ChildWatch.add (this.pid, this.cmd_watch);

        this.loop = new MainLoop (null, false);
        Gdk.threads_leave ();
        this.loop.run ();
        Gdk.threads_enter ();
        this.loop = null;
      }
      catch (Error e)
      {
        warning (e.message);
        this.running = false;
        this.status = -1;
        executable.cmd_end (this.status, 0.0);
      }
    }
    else
    {
      warning (_("Already running!"));
    }
    return this.status;
  }

  /**
   * Stop the current executable
   *
   * @throws ExecutableError If no executable running
   */
  public void stop () throws ExecutableError
  {
    if (this.running)
    {
      pid_kill (this.pid);
    }
    else
    {
      throw new ExecutableError.NO_EXECUTABLE (_("No executable launched!"));
    }
  }
}

