/* document-message.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * A message widget for Valide.Document
 */
public class Valide.DocumentMessage : Gtk.InfoBar
{
  private Gtk.Image image;
  private Gtk.Label primary_label;
  private Gtk.Label secondary_label;

  /**
   * The attached document
   */
  public Document document
  {
    construct;
    get;
  }

  /**
   * Create a new DocumentMessage
   *
   * @param document The attached document
   */
  public DocumentMessage (Document document)
  {
    Object (document: document);
  }

  construct
  {
    string text;
    Gtk.HBox hbox;
    Gtk.VBox vbox;
    Gtk.Container content_area;

    content_area = this.get_content_area () as Gtk.Container;
    hbox = new Gtk.HBox (false, 5);
    content_area.add (hbox);

    this.image = new Gtk.Image ();
    this.notify["message-type"].connect (() => {
      string stock_id;

      switch (this.message_type)
      {
        case Gtk.MessageType.INFO:
          stock_id = Gtk.Stock.DIALOG_INFO;
        break;
        case Gtk.MessageType.WARNING:
          stock_id = Gtk.Stock.DIALOG_WARNING;
        break;
        case Gtk.MessageType.QUESTION:
          stock_id = Gtk.Stock.DIALOG_QUESTION;
        break;
        case Gtk.MessageType.ERROR:
          stock_id = Gtk.Stock.DIALOG_ERROR;
        break;
        case Gtk.MessageType.OTHER:
        default:
          stock_id = Gtk.Stock.DISCARD;
        break;
      }
      this.image.set_from_stock (stock_id, Gtk.IconSize.DIALOG);
    });
    hbox.pack_start (this.image, false, true, 0);

    vbox = new Gtk.VBox (false, 0);
    hbox.pack_start (vbox, true, true, 0);

    this.primary_label = new Gtk.Label ("");
    this.primary_label.set_alignment (0, 0.5f);
    this.primary_label.set_justify (Gtk.Justification.LEFT);
    this.primary_label.set_line_wrap (true);
    vbox.pack_start (this.primary_label, true, true, 0);

    this.secondary_label = new Gtk.Label ("");
    this.secondary_label.set_alignment (0, 0.5f);
    this.secondary_label.set_justify (Gtk.Justification.LEFT);
    this.secondary_label.set_line_wrap (true);
    vbox.pack_start (this.secondary_label, false, true, 0);

    this.add_button (Gtk.Stock.REFRESH, Gtk.ResponseType.ACCEPT);
    this.add_button (Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL);
    this.response.connect ((rid) => {
      this.document.grab_focus ();
      this.document.state = DocumentState.NORMAL;
      if (rid == Gtk.ResponseType.ACCEPT)
      {
        this.document.reload ();
      }
      else
      {
        this.document.is_save = false;
      }
      this.destroy ();
    });

    switch (this.document.state)
    {
      case DocumentState.EXTERNALLY_MODIFIED:
        text = _("The file %s changed on disk.").printf (this.document.path);
        this.primary_label.set_text (text);

        if (!this.document.is_save)
        {
          text = _("Do you want to drop your changes and reload the file?");
        }
        else
        {
          text = _("Do you want to reload the file?");
        }
        this.secondary_label.set_text (text);
        this.set_message_type (Gtk.MessageType.WARNING);
      break;
      case DocumentState.BAD_ENCODING:
        text = _("Couldn't open the file %s.").printf (this.document.path);
        this.primary_label.set_text (text);

        text=_("Val(a)IDE has not been able to detect the character coding.\nPlease check that you are not trying to open a binary file.");
        this.secondary_label.set_text (text);
        this.set_message_type (Gtk.MessageType.ERROR);
      break;
    }
  }
}

