/* abstract-config-manager.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * Abstract class for manage configation files.
 */
public abstract class Valide.AbstractConfigManager : Object
{
  private static string _tmp_dir;

  /**
   * Create and show preferences dialog
   *
   * @param window The main window of the application
   */
  public abstract void preferences (Window window);

  /**
   * Function call at first launch
   */
  protected abstract void first_run ();

  /**
   * Path of the configuration file
   */
  public abstract string get_config_file();

  /**
   * Path of the user directory.
   */
  public string get_home_dir()
  {
      string path;

      path = Path.build_filename (Environment.get_home_dir (),
                                  ".%s".printf (Config.PACKAGE_NAME));
      if (!FileUtils.test (path, FileTest.EXISTS))
      {
        DirUtils.create (path, 0700);
      }
      return path;
  }

  /**
   * Path of the user config directory.
   */
  public string get_config_dir()
  {
      string path;

      path = Path.build_filename (Environment.get_user_config_dir (),
                                  Config.PACKAGE_NAME);
      if (!FileUtils.test (path, FileTest.EXISTS))
      {
        DirUtils.create (path, 0700);
      }
      return path;
  }

  /**
   * Path of the temporary directory.
   */
  public string get_tmp_dir()
  {
      if (this._tmp_dir == null)
      {
        this._tmp_dir = Path.build_filename (Environment.get_tmp_dir (),
                                             Config.PACKAGE_NAME);
        DirUtils.create_with_parents (this._tmp_dir, 0700);
      }
      return this._tmp_dir;
  }

  private string create_file (string path, string contents)
  {
    if (!FileUtils.test (path, FileTest.EXISTS))
    {
      FileStream stream = FileStream.open (path, "w");
      if (stream != null)
      {
        stream.printf (contents);
      }
      else
      {
        warning (_("Can't create %s"), path);
      }
    }
    return path;
  }

  /**
   * Create configuration file if doesn't exist, in user configuration
   * directory #home.
   *
   * @param filename a name of file.
   * @param contents a content of file.
   *
   * @return A path of file.
   */
  protected string create_preferences_file (string filename, string contents)
  {
    string path = Path.build_filename (this.get_config_dir (), filename);

    return this.create_file (path, contents);
  }

  construct
  {
    if (!FileUtils.test (this.get_home_dir (), FileTest.EXISTS))
    {
      DirUtils.create (this.get_home_dir (), 0700);
      this.first_run ();
    }
  }

  /**
   * Return interger from configuration file.
   *
   * @param group a group name.
   * @param key a key.
   *
   * @return the value associated with the key as an integer, or 0 if the key
   *         was not found or could not be parsed.
   */
  public int get_integer (string group, string key)
  {
    int ret = 0;

    try
    {
      KeyFile key_file = new KeyFile ();
      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      ret = key_file.get_integer (group, key);
    }
    catch (Error e)
    {
      debug (_("Can't read config.ini file: %s"), e.message);
    }
    return ret;
  }

  /**
   * Return string from configuration file.
   *
   * @param group a group name.
   * @param key a key.
   *
   * @return the value associated with the key as a string, or empty string if
   *         the key was not found or could not be parsed.
   */
  public string get_string (string group, string key)
  {
    string ret = "";

    try
    {
      KeyFile key_file = new KeyFile ();
      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      ret = key_file.get_value (group, key);
    }
    catch (Error e)
    {
      debug (_("Can't read config.ini file: %s"), e.message);
    }
    return ret;
  }

  /**
   * Return boolean from configuration file.
   *
   * @param group a group name.
   * @param key a key.
   *
   * @return the value associated with the key as a boolean, or false if the key
   *         was not found or could not be parsed.
   */
  public bool get_boolean (string group, string key)
  {
    bool ret = false;

    try
    {
      KeyFile key_file = new KeyFile ();
      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      ret = key_file.get_boolean (group, key);
    }
    catch (Error e)
    {
      debug (_("Can't read config.ini file: %s"), e.message);
    }
    return ret;
  }

  /**
   * Associates a new integer value with #key under #group. If key cannot be
   * found then it is created.
   *
   * @param group a group name.
   * @param key a key.
   * @param val an integer value.
   */
  public void set_integer (string group, string key, int val)
  {
    try
    {
      KeyFile key_file = new KeyFile ();

      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      key_file.set_integer (group, key, val);
      FileUtils.set_contents (this.get_config_file(), key_file.to_data (null), -1);
    }
    catch (Error e)
    {
      critical (_("Can't read config.ini file: %s"), e.message);
    }
  }

  /**
   * Associates a new string value with #key under #group. If key cannot be
   * found then it is created.
   *
   * @param group a group name.
   * @param key a key.
   * @param val a string value.
   */
  public void set_string (string group, string key, string val)
  {
    try
    {
      KeyFile key_file = new KeyFile ();

      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      key_file.set_string (group, key, val);
      FileUtils.set_contents (this.get_config_file(), key_file.to_data (null), -1);
    }
    catch (Error e)
    {
      critical (_("Can't read config.ini file: %s"), e.message);
    }
  }

  /**
   * Associates a new boolean value with #key under #group. If key cannot be
   * found then it is created.
   *
   * @param group a group name.
   * @param key a key.
   * @param val a boolean value.
   */
  public void set_boolean (string group, string key, bool val)
  {
    try
    {
      KeyFile key_file = new KeyFile ();

      key_file.load_from_file (this.get_config_file(), KeyFileFlags.NONE);
      key_file.set_boolean (group, key, val);
      FileUtils.set_contents (this.get_config_file(), key_file.to_data (null), -1);
    }
    catch (Error e)
    {
      critical (_("Can't read config.ini file: %s"), e.message);
    }
  }
}

