/*
 * ctags-vala.vala
 *
 * Copyright 2008-2010 Abderrahim Kitouni <a.kitouni@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

public class CTagsEntry {
  public int line_number;
  public string name;
  public string kind_name;
  public char kind;
  public string access;
  public string implementation;
  public string inheritance;
  public string signature;
  public string typeref;
}

[CCode (has_target = false)]
public delegate void CTagsEntryMaker (CTagsEntry entry);

private class Report : Vala.Report
{
  public override void err (Vala.SourceReference? source, string message)
  {
  }

  public override void warn (Vala.SourceReference? source, string message)
  {
  }
}

public class CTagsVisitor : Vala.CodeVisitor {
  Vala.Parser vala_parser;
  Vala.Genie.Parser genie_parser;
  List<CTagsEntry> taglist;

  public CTagsVisitor () {
    vala_parser = new Vala.Parser();
    genie_parser = new Vala.Genie.Parser();
  }

  /* helper functions */
  static string get_access (Vala.Symbol sym) {
    switch (sym.access) {
    case Vala.SymbolAccessibility.PRIVATE : return "private";
    case Vala.SymbolAccessibility.INTERNAL : return "internal";
    case Vala.SymbolAccessibility.PROTECTED : return "protected";
    case Vala.SymbolAccessibility.PUBLIC : return "public";
    }
    assert_not_reached ();
  }

  static string to_string (Vala.Iterable<Vala.DataType> seq, string sep = ",") {
    var str = new StringBuilder();
    var first = true;
    foreach (var type in seq) {
      if(first) {
        first = false;
      } else {
        str.append(sep);
      }
      str.append(type.to_string());
    }
    return str.str;
  }

  static string? implementation (dynamic Vala.Symbol sym) {
    assert((sym is Vala.Class) || (sym is Vala.Signal)
      || (sym is Vala.Property) || (sym is Vala.Method)
      || (sym is Vala.Delegate) || (sym is Vala.Constructor)
      || (sym is Vala.Destructor) || (sym is Vala.Namespace));
    var list = new List<string>();
/*
    if (!(sym is Vala.Signal) && (bool)sym.is_abstract) {
      list.append("abstract");
    }
    if (!(sym is Class) && (bool)sym.is_virtual) {
      list.append("virtual");
    }
    if (list == null) {
      return null;
    }
*/
    var ret = new StringBuilder();
    var first = true;
    foreach (var str in list) {
      if(first) {
        first = false;
      } else {
        ret.append(",");
      }
      ret.append(str);
    }
    return ret.str;
  }

  static string name (dynamic Vala.Symbol sym) {
    string name;

    if (CTags.Option.include.qualified_tags) {
      name = sym.get_full_name ();
    } else {
      if (sym is Vala.Constructor) {
        name = "construct";
      } else if (sym is Vala.Destructor) {
        name = "~" + ((Vala.Destructor)sym).this_parameter.variable_type.to_string ();
      } else {
        name = sym.name;
      }
    }
    return name;
  }

  static string signature (Vala.List<Vala.Parameter> parameter) {
    var ret = new StringBuilder("(");
    var first = true;
    foreach (var p in parameter) {
      if(first) {
        first = false;
      } else {
        ret.append(",");
      }
      if (p.ellipsis) {
        ret.append("...");
      } else {
        ret.append (p.variable_type.to_string());
        ret.append (" ");
        ret.append (p.name);
      }
    }
    ret.append (")");
    return ret.str;
  }

  static void print_tag(CTagsEntry en) {
    stdout.printf("%s: %s at %d\n", en.name, en.kind_name, en.line_number);
  }

  public override void visit_source_file (Vala.SourceFile source_file) {
    source_file.accept_children (this);
  }

  public override void visit_class (Vala.Class cl) {
    var entry = new CTagsEntry();

    entry.line_number = cl.source_reference.first_line;
    entry.name = name (cl);
    entry.kind_name = "class";
    entry.kind = 'c';
    entry.access = get_access (cl);
    entry.implementation = implementation(cl);
    if (cl.base_class != null)
    {
      entry.inheritance = name (cl.base_class);
    }
    else
    {
      entry.inheritance = to_string(cl.get_base_types(), ",");
    }

    taglist.append(entry);
//    print_tag(entry);
    cl.accept_children(this);
  }

  public override void visit_struct (Vala.Struct st) {
    var entry = new CTagsEntry();
    entry.line_number = st.source_reference.first_line;
    entry.name = name (st);
    entry.kind_name = "struct";
    entry.kind = 's';
    entry.access = get_access (st);

    taglist.append(entry);
//    print_tag(entry);
    st.accept_children(this);
  }

  public override void visit_interface (Vala.Interface iface) {
    var entry = new CTagsEntry();

    entry.line_number = iface.source_reference.first_line;
    entry.name = name (iface);
    entry.kind_name = "interface";
    entry.kind = 'i';
    entry.access = get_access (iface);
    entry.inheritance = to_string(iface.get_prerequisites());

    taglist.append(entry);
//    print_tag(entry);
    iface.accept_children(this);
  }

  public override void visit_enum (Vala.Enum en) {
    var entry = new CTagsEntry();

    entry.line_number = en.source_reference.first_line;
    entry.name = name (en);
    entry.kind_name = "enum";
    entry.kind = 'e';
    entry.access = get_access (en);

    taglist.append(entry);
//    print_tag(entry);
    en.accept_children(this);
  }

  public override void visit_error_domain (Vala.ErrorDomain edomain) {
    var entry = new CTagsEntry();

    entry.line_number = edomain.source_reference.first_line;
    entry.name = name (edomain);
    entry.kind_name = "errordomain";
    entry.kind = 'E';
    entry.access = get_access (edomain);

    taglist.append(entry);
//    print_tag(entry);
    edomain.accept_children(this);
  }

  public override void visit_enum_value (Vala.EnumValue ev) {
    var entry = new CTagsEntry();

    entry.line_number = ev.source_reference.first_line;
    entry.name = name (ev);
    entry.kind_name = "enumvalue";
    entry.kind = 'v';
    entry.access = get_access (ev);

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_error_code (Vala.ErrorCode ecode) {
    var entry = new CTagsEntry();

    entry.line_number = ecode.source_reference.first_line;
    entry.name = name (ecode);
    entry.kind_name = "errorcode";
    entry.kind = 'r';
    entry.access = get_access (ecode);

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_delegate (Vala.Delegate d) {
    var entry = new CTagsEntry();

    entry.line_number = d.source_reference.first_line;
    entry.name = name (d);
    entry.kind_name = "delegate";
    entry.kind = 'd';
    entry.access = get_access (d);
    entry.implementation = implementation(d);
    entry.typeref = d.return_type.to_qualified_string();
    entry.signature = signature(d.get_parameters());

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_signal (Vala.Signal sig) {
    var entry = new CTagsEntry();

    entry.line_number = sig.source_reference.first_line;
    entry.name = name (sig);
    entry.kind_name = "signal";
    entry.kind = 'S';
    entry.access = get_access (sig);
    entry.implementation = implementation(sig);
    entry.typeref = sig.return_type.to_qualified_string();
    entry.signature = signature(sig.get_parameters());

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_field (Vala.Field f) {
    var entry = new CTagsEntry();

    entry.line_number = f.source_reference.first_line;
    entry.name = name (f);
    entry.kind_name = "field";
    entry.kind = 'f';
    entry.access = get_access (f);
    entry.typeref = f.variable_type.to_string();

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_constant (Vala.Constant c) {
    var entry = new CTagsEntry();

    entry.line_number = c.source_reference.first_line;
    entry.name = name (c);
    entry.kind_name = "field";
    entry.kind = 'f';
    entry.access = get_access (c);
    entry.typeref = c.type_reference.to_qualified_string();

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_property (Vala.Property prop) {
    var entry = new CTagsEntry();

    entry.line_number = prop.source_reference.first_line;
    entry.name = name (prop);
    entry.kind_name = "property";
    entry.kind = 'p';
    entry.access = get_access (prop);
    entry.implementation = implementation(prop);
    entry.typeref = prop.property_type.to_qualified_string();

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_method (Vala.Method m) {
    var entry = new CTagsEntry();

    entry.line_number = m.source_reference.first_line;
    entry.name = name (m);
    entry.kind_name = "method";
    entry.kind = 'm';
    entry.access = get_access (m);
    entry.implementation = implementation(m);
    entry.typeref = m.return_type.to_qualified_string();
    entry.signature = signature(m.get_parameters());

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_namespace (Vala.Namespace m) {
    if (m.name != null)
    {
      var entry = new CTagsEntry();

      entry.line_number = m.source_reference.first_line;
      entry.name = name (m);
      entry.kind_name = "namespace";
      entry.kind = 'M';
      entry.access = get_access (m);

      taglist.append(entry);
//      print_tag(entry);
    }
  }

  public override void visit_creation_method (Vala.CreationMethod m) {
    var entry = new CTagsEntry();

    entry.line_number = m.source_reference.first_line;
    entry.name = name (m);
    entry.kind_name = "method";
    entry.kind = 'm';
    entry.access = get_access (m);
    entry.implementation = implementation(m);
    entry.typeref = "";
    entry.signature = signature(m.get_parameters());

    taglist.append(entry);
//    print_tag(entry);
  }
/*
  public override void visit_constructor (Constructor c) {
    var entry = new CTagsEntry();

    entry.line_number = c.source_reference.first_line;
    entry.name = name (c);
    entry.kind_name = "method";
    entry.kind = 'm';
    entry.access = get_access (c);
    entry.implementation = implementation(c);
    entry.typeref = "";
    entry.signature = "construct";

    taglist.append(entry);
//    print_tag(entry);
  }

  public override void visit_destructor (Destructor d) {
    var entry = new CTagsEntry();

    entry.line_number = d.source_reference.first_line;
    entry.name = name (d);
    entry.kind_name = "method";
    entry.kind = 'm';
    entry.access = "";
    entry.implementation = "";
    entry.typeref = "";
    entry.signature = "";

    taglist.append(entry);
//    print_tag(entry);
  }
*/
  public override void visit_local_variable (Vala.LocalVariable local) {
    var entry = new CTagsEntry();

    entry.line_number = local.source_reference.first_line;
    entry.name = name (local);
    entry.kind_name = "local";
    entry.kind = 'l';
    entry.access = get_access (local);

    taglist.append(entry);
//    print_tag(entry);
  }

  public void parse_vala (string filename, CTagsEntryMaker maker ) {
    /**
     * @bug CodeVisitor doesn't derive from GObject (r1757), construct doesn't call.
     */
    vala_parser = new Vala.Parser();
    taglist = new List<CTagsEntry>();
    /* We create a context for every source file so that Parser.parse(context)
     * don't parse a file multiple times causing errors. Parser.parse_file(source_file)
     * assumes that Parser.context is the same as source_file.context anyway */
    var context = new Vala.CodeContext();
    context.report = new Report ();
    var source_type = Vala.SourceFileType.SOURCE;
    if (filename.has_suffix("vapi"))
    {
      source_type = Vala.SourceFileType.PACKAGE;
    }
    else
    {
      source_type = Vala.SourceFileType.SOURCE;
    }
    var source_file = new Vala.SourceFile(context, source_type, filename);
    context.add_source_file(source_file);
    Vala.CodeContext.push (context);
    vala_parser.parse(context);
    context.accept(this);
    foreach (var tagentry in taglist) {
      maker(tagentry);
    }
    taglist = null;
  }

  public void parse_genie (string filename, CTagsEntryMaker maker ) {
    /**
     * @bug CodeVisitor doesn't derive from GObject (r1757), construct doesn't call.
     */
    genie_parser = new Vala.Genie.Parser();
    taglist = new List<CTagsEntry>();
    var context = new Vala.CodeContext();
    context.report = new Report ();
    var source_type = Vala.SourceFileType.SOURCE;
    if (filename.has_suffix("vapi"))
    {
      source_type = Vala.SourceFileType.PACKAGE;
    }
    else
    {
      source_type = Vala.SourceFileType.SOURCE;
    }
    var source_file = new Vala.SourceFile(context, source_type, filename);
    context.add_source_file(source_file);
    Vala.CodeContext.push (context);
    genie_parser.parse(context);
    context.accept(this);
    foreach (var tagentry in taglist) {
      maker(tagentry);
    }
    taglist = null;
  }
}

