// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_DATA_REDUCTION_PROXY_BROWSER_DATA_REDUCTION_PROXY_PARAMS_H_
#define COMPONENTS_DATA_REDUCTION_PROXY_BROWSER_DATA_REDUCTION_PROXY_PARAMS_H_

#include <string>
#include <utility>
#include <vector>

#include "base/macros.h"
#include "net/base/host_port_pair.h"
#include "url/gurl.h"

namespace net {
class URLRequest;
}

namespace data_reduction_proxy {
// Provides initialization parameters. Proxy origins, the probe url, and the
// authentication key are taken from flags if available and from preprocessor
// constants otherwise. Only the key may be changed after construction. The
// DataReductionProxySettings class and others use this class to determine
// the necessary DNS names and keys to configure use of the data reduction
// proxy.
class DataReductionProxyParams {
 public:
  static const unsigned int kAllowed = (1 << 0);
  static const unsigned int kFallbackAllowed = (1 << 1);
  static const unsigned int kAlternativeAllowed = (1 << 2);
  static const unsigned int kPromoAllowed = (1 << 3);

  typedef std::vector<GURL> DataReductionProxyList;

  // Returns true if this client is part of the data reduction proxy field
  // trial.
  static bool IsIncludedInFieldTrial();

  // Returns true if this client is part of field trial to use an alternative
  // configuration for the data reduction proxy.
  static bool IsIncludedInAlternativeFieldTrial();

  // Returns true if this client is part of the field trial that should display
  // a promotion for the data reduction proxy.
  static bool IsIncludedInPromoFieldTrial();

  // Returns true if this client is part of a field trial that uses preconnect
  // hinting.
  static bool IsIncludedInPreconnectHintingFieldTrial();

  // Returns true if the authentication key was set on the command line.
  static bool IsKeySetOnCommandLine();

  // Constructs configuration parameters. If |kAllowed|, then the standard
  // data reduction proxy configuration is allowed to be used. If
  // |kfallbackAllowed| a fallback proxy can be used if the primary proxy is
  // bypassed or disabled. If |kAlternativeAllowed| then an alternative proxy
  // configuration is allowed to be used. This alternative configuration would
  // replace the primary and fallback proxy configurations if enabled. Finally
  // if |kPromoAllowed|, the client may show a promotion for the data reduction
  // proxy.
  //
  // A standard configuration has a primary proxy, and a fallback proxy for
  // HTTP traffic. The alternative configuration has a different primary and
  // fallback proxy for HTTP traffic, and an SSL proxy.

  DataReductionProxyParams(int flags);

  virtual ~DataReductionProxyParams();

  // Returns true if a data reduction proxy was used for the given |request|.
  // If true, |proxy_servers.first| will contain the name of the proxy that was
  // used. |proxy_servers.second| will contain the name of the data reduction
  // proxy server that would be used if |proxy_server.first| is bypassed, if one
  // exists. |proxy_servers| can be NULL if the caller isn't interested in its
  // values.
  virtual bool WasDataReductionProxyUsed(
      const net::URLRequest* request,
      std::pair<GURL, GURL>* proxy_servers) const;

  // Returns true if the specified |host_port_pair| matches a data reduction
  // proxy. If true, |proxy_servers.first| will contain the name of the proxy
  // that matches. |proxy_servers.second| will contain the name of the
  // data reduction proxy server that would be used if |proxy_server.first| is
  // bypassed, if one exists. |proxy_servers| can be NULL if the caller isn't
  // interested in its values.
  bool IsDataReductionProxy(const net::HostPortPair& host_port_pair,
                            std::pair<GURL, GURL>* proxy_servers) const;

  // Returns true if this request will be sent through the data request proxy
  // based on applying the param rules to the URL. We do not check bad proxy
  // list.
  virtual bool IsDataReductionProxyEligible(const net::URLRequest* request);

  // Returns the data reduction proxy primary origin.
  const GURL& origin() const {
    return origin_;
  }

  // Returns the data reduction proxy fallback origin.
  const GURL& fallback_origin() const {
    return fallback_origin_;
  }

  // Returns the data reduction proxy ssl origin that is used with the
  // alternative proxy configuration.
  const GURL& ssl_origin() const {
    return ssl_origin_;
  }

  // Returns the alternative data reduction proxy primary origin.
  const GURL& alt_origin() const {
    return alt_origin_;
  }

  // Returns the alternative data reduction proxy fallback origin.
  const GURL& alt_fallback_origin() const {
    return alt_fallback_origin_;
  }

  // Returns the URL to probe to decide if the primary origin should be used.
  const GURL& probe_url() const {
    return probe_url_;
  }

  // Returns the URL to fetch to warm the data reduction proxy connection.
  const GURL& warmup_url() const {
    return warmup_url_;
  }

  // Set the proxy authentication key.
  void set_key(const std::string& key) {
    key_ = key;
  }

  // Returns the proxy authentication key.
  const std::string& key() const {
    return key_;
  }

  // Returns true if the data reduction proxy configuration may be used.
  bool allowed() const {
    return allowed_;
  }

  // Returns true if the fallback proxy may be used.
  bool fallback_allowed() const {
    return fallback_allowed_;
  }

  // Returns true if the alternative data reduction proxy configuration may be
  // used.
  bool alternative_allowed() const {
    return alt_allowed_;
  }

  // Returns true if the data reduction proxy promo may be shown.
  // This is idependent of whether the data reduction proxy is allowed.
  // TODO(bengr): maybe tie to whether proxy is allowed.
  bool promo_allowed() const {
    return promo_allowed_;
  }

  // Given |allowed_|, |fallback_allowed_|, and |alt_allowed_|, returns the
  // list of data reduction proxies that may be used.
  DataReductionProxyList GetAllowedProxies() const;

 protected:
  // Test constructor that optionally won't call Init();
  DataReductionProxyParams(int flags,
                           bool should_call_init);

  // Initialize the values of the proxies, probe URL, and key from command
  // line flags and preprocessor constants, and check that there are
  // corresponding definitions for the allowed configurations.
  bool Init(bool allowed, bool fallback_allowed, bool alt_allowed);

  // Initialize the values of the proxies, probe URL, and key from command
  // line flags and preprocessor constants.
  void InitWithoutChecks();

  // Returns the corresponding string from preprocessor constants if defined,
  // and an empty string otherwise.
  virtual std::string GetDefaultKey() const;
  virtual std::string GetDefaultDevOrigin() const;
  virtual std::string GetDefaultOrigin() const;
  virtual std::string GetDefaultFallbackOrigin() const;
  virtual std::string GetDefaultSSLOrigin() const;
  virtual std::string GetDefaultAltOrigin() const;
  virtual std::string GetDefaultAltFallbackOrigin() const;
  virtual std::string GetDefaultProbeURL() const;
  virtual std::string GetDefaultWarmupURL() const;

 private:
  GURL origin_;
  GURL fallback_origin_;
  GURL ssl_origin_;
  GURL alt_origin_;
  GURL alt_fallback_origin_;
  GURL probe_url_;
  GURL warmup_url_;

  std::string key_;

  bool allowed_;
  const bool fallback_allowed_;
  bool alt_allowed_;
  const bool promo_allowed_;

  DISALLOW_COPY_AND_ASSIGN(DataReductionProxyParams);
};

}  // namespace data_reduction_proxy
#endif  // COMPONENTS_DATA_REDUCTION_PROXY_BROWSER_DATA_REDUCTION_PROXY_PARAMS_H_
