// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_LOGIN_ENROLLMENT_ENROLLMENT_SCREEN_H_
#define CHROME_BROWSER_CHROMEOS_LOGIN_ENROLLMENT_ENROLLMENT_SCREEN_H_

#include <string>

#include "base/basictypes.h"
#include "base/compiler_specific.h"
#include "base/gtest_prod_util.h"
#include "base/memory/weak_ptr.h"
#include "chrome/browser/chromeos/login/enrollment/enrollment_screen_actor.h"
#include "chrome/browser/chromeos/login/screens/wizard_screen.h"
#include "components/policy/core/common/cloud/cloud_policy_constants.h"

namespace chromeos {

class ScreenObserver;

// The screen implementation that links the enterprise enrollment UI into the
// OOBE wizard.
class EnrollmentScreen
    : public WizardScreen,
      public EnrollmentScreenActor::Controller {
 public:
  EnrollmentScreen(ScreenObserver* observer,
                   EnrollmentScreenActor* actor);
  virtual ~EnrollmentScreen();

  void SetParameters(EnrollmentScreenActor::EnrollmentMode enrollment_mode,
                     const std::string& management_domain,
                     const std::string& enrollment_user);

  // WizardScreen implementation:
  virtual void PrepareToShow() OVERRIDE;
  virtual void Show() OVERRIDE;
  virtual void Hide() OVERRIDE;
  virtual std::string GetName() const OVERRIDE;

  // EnrollmentScreenActor::Controller implementation:
  virtual void OnLoginDone(const std::string& user) OVERRIDE;
  virtual void OnAuthError(const GoogleServiceAuthError& error) OVERRIDE;
  virtual void OnOAuthTokenAvailable(const std::string& oauth_token) OVERRIDE;
  virtual void OnRetry() OVERRIDE;
  virtual void OnCancel() OVERRIDE;
  virtual void OnConfirmationClosed() OVERRIDE;

  // Used for testing.
  EnrollmentScreenActor* GetActor() {
    return actor_;
  }

 private:
  FRIEND_TEST_ALL_PREFIXES(EnrollmentScreenTest, TestSuccess);

  // Starts the Lockbox storage process.
  void WriteInstallAttributesData();

  // Kicks off the policy infrastructure to register with the service.
  void RegisterForDevicePolicy(const std::string& token);

  // Handles enrollment completion. Logs a UMA sample and requests the actor to
  // show the specified enrollment status.
  void ReportEnrollmentStatus(policy::EnrollmentStatus status);

  // Shows successful enrollment status after all enrollment related file
  // operations are completed.
  void ShowEnrollmentStatusOnSuccess(const policy::EnrollmentStatus& status);

  // Logs a UMA event in the kMetricEnrollment histogram. If auto-enrollment is
  // on |sample| is ignored and a kMetricEnrollmentAutoFailed sample is logged
  // instead.
  void UMAFailure(int sample);

  // Shows the signin screen. Used as a callback to run after auth reset.
  void ShowSigninScreen();

  // Convenience helper to check for auto enrollment mode.
  bool is_auto_enrollment() const {
    return enrollment_mode_ == EnrollmentScreenActor::ENROLLMENT_MODE_AUTO;
  }

  EnrollmentScreenActor* actor_;
  EnrollmentScreenActor::EnrollmentMode enrollment_mode_;
  bool enrollment_failed_once_;
  std::string user_;
  int lockbox_init_duration_;
  base::WeakPtrFactory<EnrollmentScreen> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(EnrollmentScreen);
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_LOGIN_ENROLLMENT_ENROLLMENT_SCREEN_H_
